package io.bitexpress.openapi.client;

import com.fasterxml.jackson.databind.ObjectMapper;
import io.bitexpress.openapi.client.sign.KeyReader;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.exception.ContextedRuntimeException;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;
import org.springframework.http.client.ClientHttpRequestFactory;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.http.converter.StringHttpMessageConverter;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;
import org.springframework.web.client.RestTemplate;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.util.Arrays;

public class GatewayClientBuilder extends BaseBuilder {
    private String url;

    private ClientHttpRequestFactory clientHttpRequestFactory;

    private Resource clientPrivateKeyResource;

    public GatewayClient build() {
        GatewayClientImpl client = new GatewayClientImpl();
        client.setUrl(Validate.notBlank(url, "url is blank"));

        ObjectMapper objectMapper = createObjectMapper();
        client.setObjectMapper(objectMapper);
        client.setRequestFactory(createRequestFactory(objectMapper));
        client.setRestTemplate(createRestTemplate(objectMapper));
        return client;
    }


    private RequestFactory createRequestFactory(ObjectMapper objectMapper) {
        RequestFactory requestFactory = new RequestFactory();
        requestFactory.setObjectMapper(objectMapper);
        requestFactory.setUid(uid);
        requestFactory.setIssuerCode(issuerCode);
        Pair<PrivateKey, PublicKey> keyPair = createKeyPair();
        requestFactory.setPrivateKey(keyPair.getLeft());
        return requestFactory;
    }

    private RestTemplate createRestTemplate(ObjectMapper objectMapper) {
        MappingJackson2HttpMessageConverter mappingJackson2HttpMessageConverter = new WriteOnlyMappingJackson2HttpMessageConverter(
                objectMapper);
        StringHttpMessageConverter stringHttpMessageConverter = new StringHttpMessageConverter();
        if (clientHttpRequestFactory == null) {
            clientHttpRequestFactory = new HttpComponentsClientHttpRequestFactory();
        }
        RestTemplate restTemplate = new RestTemplate(clientHttpRequestFactory);
        restTemplate
                .setMessageConverters(Arrays.asList(mappingJackson2HttpMessageConverter, stringHttpMessageConverter));
        return restTemplate;
    }

    private Pair<PrivateKey, PublicKey> createKeyPair() {
        try (InputStream is = clientPrivateKeyResource.getInputStream()) {
            String keyPairPem = IOUtils.toString(is, StandardCharsets.US_ASCII);
            return keyReader.readKeyPair(keyPairPem);
        } catch (IOException e) {
            throw new ContextedRuntimeException(e);
        }
    }


    public void setClientPrivateKeyFilePath(Path clientPrivateKeyFilePath) {
        setClientPrivateKeyResource(new FileSystemResource(clientPrivateKeyFilePath));
    }

    public void setClientPrivateKeyResource(Resource clientPrivateKeyResource) {
        this.clientPrivateKeyResource = clientPrivateKeyResource;
    }

    public void setUrl(String url) {
        this.url = url;
    }


    public void setUid(Long uid) {
        this.uid = uid;
    }

    public void setIssuerCode(String issuerCode) {
        this.issuerCode = issuerCode;
    }

    public void setClientHttpRequestFactory(ClientHttpRequestFactory clientHttpRequestFactory) {
        this.clientHttpRequestFactory = clientHttpRequestFactory;
    }

    public void setKeyReader(KeyReader keyReader) {
        this.keyReader = keyReader;
    }

}
