/*
 * This file is part of the BigConnect project.
 *
 * Copyright (c) 2013-2020 MWARE SOLUTIONS SRL
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License version 3
 * as published by the Free Software Foundation with the addition of the
 * following permission added to Section 15 as permitted in Section 7(a):
 * FOR ANY PART OF THE COVERED WORK IN WHICH THE COPYRIGHT IS OWNED BY
 * MWARE SOLUTIONS SRL, MWARE SOLUTIONS SRL DISCLAIMS THE WARRANTY OF
 * NON INFRINGEMENT OF THIRD PARTY RIGHTS

 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Affero General Public License for more details.
 * You should have received a copy of the GNU Affero General Public License
 * along with this program; if not, see http://www.gnu.org/licenses or write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA, 02110-1301 USA, or download the license from the following URL:
 * https://www.gnu.org/licenses/agpl-3.0.txt
 *
 * The interactive user interfaces in modified source and object code versions
 * of this program must display Appropriate Legal Notices, as required under
 * Section 5 of the GNU Affero General Public License.
 *
 * You can be released from the requirements of the license by purchasing
 * a commercial license. Buying such a license is mandatory as soon as you
 * develop commercial activities involving the BigConnect software without
 * disclosing the source code of your own applications.
 *
 * These activities include: offering paid services to customers as an ASP,
 * embedding the product in a web application, shipping BigConnect with a
 * closed source product.
 */
package com.mware.ge.store;

import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;

public class EdgesWithEdgeInfo extends Edges {
    // We used to use a HashMap here but that was too slow. Replaced with a List since we really don't care about duplicates anyway.
    private final List<Map.Entry<String, StorableEdgeInfo>> pairs = new ArrayList<>();

    public void add(String edgeId, StorableEdgeInfo edgeInfo) {
        pairs.add(new Pair(edgeId, edgeInfo));
    }

    public void remove(String edgeId) {
        int i = indexOf(edgeId);
        if (i >= 0) {
            pairs.remove(i);
        }
    }

    private int indexOf(String edgeId) {
        for (int i = 0; i < pairs.size(); i++) {
            if (pairs.get(i).getKey().equals(edgeId)) {
                return i;
            }
        }
        return -1;
    }

    public void clear() {
        pairs.clear();
    }

    public StorableEdgeInfo get(String edgeId) {
        int i = indexOf(edgeId);
        if (i >= 0) {
            return pairs.get(i).getValue();
        }
        return null;
    }

    public Iterable<StorableEdgeInfo> getEdgeInfos() {
        return () -> {
            final Iterator<Map.Entry<String, StorableEdgeInfo>> it = pairs.iterator();
            return new Iterator<StorableEdgeInfo>() {
                @Override
                public boolean hasNext() {
                    return it.hasNext();
                }

                @Override
                public StorableEdgeInfo next() {
                    return it.next().getValue();
                }

                @Override
                public void remove() {
                    throw new RuntimeException("not supported");
                }
            };
        };
    }

    public Iterable<Map.Entry<String, StorableEdgeInfo>> getEntries() {
        return pairs;
    }

    public EdgesWithCount getEdgesWithCount() {
        EdgesWithCount edgesWithCount = new EdgesWithCount();
        Map<String, AtomicInteger> edgesByLabels = new HashMap<>();
        for (Map.Entry<String, StorableEdgeInfo> edgeEntry : getEntries()) {
            String label = edgeEntry.getValue().getLabel();
            edgesByLabels.computeIfAbsent(label, k -> new AtomicInteger(0))
                    .incrementAndGet();
        }
        edgesByLabels.forEach((k, v) -> edgesWithCount.add(k, v.get()));
        return edgesWithCount;
    }

    private static class Pair implements Map.Entry<String, StorableEdgeInfo> {
        private final String edgeId;
        private final StorableEdgeInfo edgeInfo;

        public Pair(String edgeId, StorableEdgeInfo edgeInfo) {
            this.edgeId = edgeId;
            this.edgeInfo = edgeInfo;
        }

        @Override
        public String getKey() {
            return edgeId;
        }

        @Override
        public StorableEdgeInfo getValue() {
            return edgeInfo;
        }

        @Override
        public StorableEdgeInfo setValue(StorableEdgeInfo value) {
            throw new RuntimeException("not supported");
        }
    }
}
