/*
 * This file is part of the BigConnect project.
 *
 * Copyright (c) 2013-2020 MWARE SOLUTIONS SRL
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License version 3
 * as published by the Free Software Foundation with the addition of the
 * following permission added to Section 15 as permitted in Section 7(a):
 * FOR ANY PART OF THE COVERED WORK IN WHICH THE COPYRIGHT IS OWNED BY
 * MWARE SOLUTIONS SRL, MWARE SOLUTIONS SRL DISCLAIMS THE WARRANTY OF
 * NON INFRINGEMENT OF THIRD PARTY RIGHTS

 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Affero General Public License for more details.
 * You should have received a copy of the GNU Affero General Public License
 * along with this program; if not, see http://www.gnu.org/licenses or write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA, 02110-1301 USA, or download the license from the following URL:
 * https://www.gnu.org/licenses/agpl-3.0.txt
 *
 * The interactive user interfaces in modified source and object code versions
 * of this program must display Appropriate Legal Notices, as required under
 * Section 5 of the GNU Affero General Public License.
 *
 * You can be released from the requirements of the license by purchasing
 * a commercial license. Buying such a license is mandatory as soon as you
 * develop commercial activities involving the BigConnect software without
 * disclosing the source code of your own applications.
 *
 * These activities include: offering paid services to customers as an ASP,
 * embedding the product in a web application, shipping BigConnect with a
 * closed source product.
 */
package io.bigconnect.dw.ner.common.extractor;

import com.bericotech.clavin.extractor.LocationOccurrence;
import com.bericotech.clavin.gazetteer.CountryCode;
import com.bericotech.clavin.gazetteer.GeoName;
import com.bericotech.clavin.resolver.ResolvedLocation;
import com.mware.core.config.Configuration;
import io.bigconnect.dw.ner.common.people.ResolvedPerson;
import io.bigconnect.dw.ner.common.places.CountryGeoNameLookup;
import io.bigconnect.dw.ner.common.orgs.ResolvedOrganization;
import java.util.ArrayList;
import java.util.List;

/**
 * Simple wrapper around results generated by an entity extractor
 *
 * @author rahulb
 */
public class ExtractedEntities {
    private List<LocationOccurrence> locations;
    private List<ResolvedLocation> resolvedLocations;

    private List<PersonOccurrence> people;
    private List<ResolvedPerson> resolvedPeople;

    private List<OrganizationOccurrence> organizations;
    private List<ResolvedOrganization> resolvedOrganizations;

    private List<GenericOccurrence> otherEntities;

    private Configuration configuration;

    public ExtractedEntities(Configuration configuration) {
        this.configuration = configuration;
        locations = new ArrayList<>();
        resolvedPeople = new ArrayList<>();
        people = new ArrayList<>();
        resolvedLocations = new ArrayList<>();
        organizations = new ArrayList<>();
        resolvedOrganizations = new ArrayList<>();
        otherEntities = new ArrayList<>();
    }

    /**
     * join two ExtractedEntities
     *
     * @param e2 - ExtractedEntities to join to this one
     */
    public void merge(ExtractedEntities e2) {
        locations.addAll(e2.getLocations());
        resolvedPeople.addAll(e2.getResolvedPeople());
        people.addAll(e2.getPeople());
        resolvedLocations.addAll(e2.getResolvedLocations());
        organizations.addAll(e2.getOrganizations());
        resolvedOrganizations.addAll(e2.getResolvedOrganizations());
        otherEntities.addAll(e2.getOtherEntities());
    }

    public void addOrganization(OrganizationOccurrence organization) {
        organizations.add(organization);
    }

    public void addPerson(PersonOccurrence person) {
        people.add(person);
    }

    public void addLocation(LocationOccurrence location) {
        locations.add(location);
    }

    public void addGenericEntity(GenericOccurrence occurrence) {
        otherEntities.add(occurrence);
    }

    public List<PersonOccurrence> getPeople() {
        return people;
    }

    public List<LocationOccurrence> getLocations() {
        return locations;
    }

    public List<OrganizationOccurrence> getOrganizations() {
        return organizations;
    }

    public List<GenericOccurrence> getOtherEntities() {
        return otherEntities;
    }

    public void setResolvedLocations(List<ResolvedLocation> locs) {
        resolvedLocations = locs;
    }

    public List<ResolvedLocation> getResolvedLocations() {
        return resolvedLocations;
    }

    public List<ResolvedOrganization> getResolvedOrganizations() {
        return resolvedOrganizations;
    }

    public List<CountryCode> getUniqueCountries() {
        return getUniqueCountries(resolvedLocations);
    }

    public List<GeoName> getUniqueCountryGeoNames() {
        return getUniqueCountryGeoNames(resolvedLocations, configuration);
    }

    public static List<CountryCode> getUniqueCountries(List<ResolvedLocation> resolvedLocations) {
        List<CountryCode> countries = new ArrayList<>();
        for (ResolvedLocation resolvedLocation : resolvedLocations) {
            CountryCode country = resolvedLocation.getGeoname().getPrimaryCountryCode();
            if (country == CountryCode.NULL) {  // skip things that aren't in countries (ie. "Asia")
                continue;
            }
            if (!countries.contains(country)) {
                countries.add(country);
            }
        }
        return countries;
    }

    public static List<GeoName> getUniqueCountryGeoNames(List<ResolvedLocation> resolvedLocations, Configuration configuration) {
        List<CountryCode> countryCodes = getUniqueCountries(resolvedLocations);
        List<GeoName> geoNames = new ArrayList<>();
        for (CountryCode countryCode : countryCodes) {
            geoNames.add(CountryGeoNameLookup.lookup(countryCode.name(), configuration));
        }
        return geoNames;
    }

    public void setResolvedPeople(List<ResolvedPerson> resolvedPeople) {
        this.resolvedPeople = resolvedPeople;
    }

    public List<ResolvedPerson> getResolvedPeople() {
        return resolvedPeople;
    }

    public void setResolvedOrganizations(List<ResolvedOrganization> resolvedOrganizations) {
        this.resolvedOrganizations = resolvedOrganizations;
    }

}
