/*
 *
 * (C) Copyright 2018 Atomic BITS (http://atomicbits.io).
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *  Contributors:
 *      Peter Rigole
 *
 */

package io.atomicbits.scraml.dsl.androidjavajackson.json;

import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;
import io.atomicbits.scraml.dsl.androidjavajackson.DateTimeRFC3339;

import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;


/**
 * Created by peter on 8/10/17.
 *
 * See: https://www.ietf.org/rfc/rfc3339.txt
 *
 */
public class DateTimeRFC3339Deserializer extends JsonDeserializer<DateTimeRFC3339> {


    @Override
    public DateTimeRFC3339 deserialize(JsonParser jp, DeserializationContext ctxt) throws IOException, JsonProcessingException {
        DateTimeRFC3339 dateTimeRFC3339 = null;
        String dateString = jp.getText();

        if (dateString != null && !dateString.isEmpty()) {

            Date date = null;

            // Keep in mind that SimpleDateFormat is NOT THREADSAFE!
            // Don't instantiate these variables on the class-level! Leave them here!
            List<SimpleDateFormat> knownPatterns = new ArrayList<SimpleDateFormat>();
            // See:
            // https://stackoverflow.com/questions/40369287/what-pattern-should-be-used-to-parse-rfc-3339-datetime-strings-in-java
            // https://stackoverflow.com/questions/4024544/how-to-parse-dates-in-multiple-formats-using-simpledateformat/4024604#4024604
            // the first three, see: play.api.libs.json.IsoDateReads
            knownPatterns.add(new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSX"));
            knownPatterns.add(new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS"));
            knownPatterns.add(new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssX"));
            // the last in the row, see:
            // https://stackoverflow.com/questions/4024544/how-to-parse-dates-in-multiple-formats-using-simpledateformat/4024604#4024604
            knownPatterns.add(new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'"));
            knownPatterns.add(new SimpleDateFormat("yyyy-MM-dd'T'HH:mm.ss'Z'"));
            knownPatterns.add(new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss"));
            knownPatterns.add(new SimpleDateFormat("yyyy-MM-dd' 'HH:mm:ss"));
            knownPatterns.add(new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssXXX"));

            for (SimpleDateFormat pattern : knownPatterns) {
                try {
                    if (date == null) {
                        date = pattern.parse(dateString);
                    }
                } catch (ParseException pe) {
                    // Loop on
                }
            }

            if (date == null) {
                throw new JsonParseException("The date " + dateString + " is not an RFC3339 date.", jp.getCurrentLocation());
            }

            dateTimeRFC3339 = new DateTimeRFC3339();
            dateTimeRFC3339.setDateTime(date);
        }

        return dateTimeRFC3339;
    }

}
