package io.architect;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.HashMap;

import org.apache.http.HttpResponse;
import org.apache.http.client.CookieStore;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.CookieSpecs;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.impl.client.BasicCookieStore;
import org.apache.http.impl.client.HttpClientBuilder;


public class ArchitectSession {
    private HttpClient _httpClient;
    private CookieStore _cookieStore;
    private HttpClientContext _httpClientContext;
    private String _baseUrl;

    public ArchitectSession(String baseUrl) {
        _baseUrl = baseUrl;

        _cookieStore = new BasicCookieStore();
        _httpClientContext = new HttpClientContext();
        _httpClientContext.setAttribute(HttpClientContext.COOKIE_STORE, _cookieStore);

        RequestConfig requestConfig = RequestConfig.custom()
            .setCookieSpec(CookieSpecs.DEFAULT)
            .setConnectionRequestTimeout(60000)
            .build();

        _httpClient = HttpClientBuilder.create()
            .setDefaultRequestConfig(requestConfig)
            .build();
    }

    private URI buildFullURI(String path, HashMap<String, String> params) {
        try {
            URIBuilder uriBuilder = new URIBuilder(_baseUrl).setPath(path);
            if (params != null) {
                for (HashMap.Entry<String, String> entry : params.entrySet()) {
                    uriBuilder.addParameter(entry.getKey(), entry.getValue());
                }
            }
            return uriBuilder.build();
        } catch(URISyntaxException ex) {
            ex.printStackTrace();
            System.exit(1);
            return null;
        }
    }

    public HttpResponse post(String path, HashMap<String, String> params) throws IOException {
        URI fullUri = buildFullURI(path, params);
        return _httpClient.execute(new HttpPost(fullUri), _httpClientContext);
    }

    public HttpResponse post(String path) throws IOException {
        URI fullUri = buildFullURI(path, null);
        return _httpClient.execute(new HttpPost(fullUri), _httpClientContext);
    }

    public HttpResponse get(String path, HashMap<String, String> params) throws IOException {
        URI fullUri = buildFullURI(path, params);
        return _httpClient.execute(new HttpGet(fullUri), _httpClientContext);
    }

    public HttpResponse get(String path) throws IOException {
        URI fullUri = buildFullURI(path, null);
        return _httpClient.execute(new HttpGet(fullUri), _httpClientContext);
    }

    public HttpResponse put(String path, HashMap<String, String> params) throws IOException {
        URI fullUri = buildFullURI(path, params);
        return _httpClient.execute(new HttpPut(fullUri), _httpClientContext);
    }

    public HttpResponse put(String path) throws IOException {
        URI fullUri = buildFullURI(path, null);
        return _httpClient.execute(new HttpPut(fullUri), _httpClientContext);
    }

    public HttpResponse delete(String path, HashMap<String, String> params) throws IOException {
        URI fullUri = buildFullURI(path, params);
        return _httpClient.execute(new HttpDelete(fullUri), _httpClientContext);
    }

    public HttpResponse delete(String path) throws IOException {
        URI fullUri = buildFullURI(path, null);
        return _httpClient.execute(new HttpDelete(fullUri), _httpClientContext);
    }
}