/*
 * Decompiled with CFR 0.152.
 */
package io.airlift.security.jwks;

import com.fasterxml.jackson.annotation.JsonAnySetter;
import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import io.airlift.json.JsonCodec;
import io.airlift.log.Logger;
import io.airlift.security.jwks.EcCurve;
import io.airlift.security.jwks.JwkEcPublicKey;
import io.airlift.security.jwks.JwkPublicKey;
import io.airlift.security.jwks.JwkRsaPublicKey;
import java.math.BigInteger;
import java.security.PublicKey;
import java.security.spec.ECParameterSpec;
import java.security.spec.ECPoint;
import java.util.Base64;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;

public final class JwksDecoder {
    private static final Logger log = Logger.get(JwksDecoder.class);
    private static final JsonCodec<JsonKeys> KEYS_CODEC = JsonCodec.jsonCodec(JsonKeys.class);

    private JwksDecoder() {
    }

    public static Map<String, PublicKey> decodeKeys(String jwksJson) {
        JsonKeys keys = (JsonKeys)KEYS_CODEC.fromJson(jwksJson);
        return (Map)keys.getKeys().stream().map(JwksDecoder::tryDecodeJwkKey).filter(Optional::isPresent).map(Optional::get).collect(ImmutableMap.toImmutableMap(JwkPublicKey::getKeyId, Function.identity()));
    }

    public static Optional<? extends JwkPublicKey> tryDecodeJwkKey(JsonKey key) {
        if (!key.getKid().isPresent() || key.getKid().get().isEmpty()) {
            return Optional.empty();
        }
        String keyId = key.getKid().get();
        switch (key.getKty()) {
            case "RSA": {
                return JwksDecoder.tryDecodeRsaKey(keyId, key);
            }
            case "EC": {
                return JwksDecoder.tryDecodeEcKey(keyId, key);
            }
        }
        return Optional.empty();
    }

    public static Optional<JwkRsaPublicKey> tryDecodeRsaKey(String keyId, JsonKey key) {
        Optional modulus = key.getStringProperty("n").flatMap(encodedModulus -> JwksDecoder.decodeBigint(keyId, "modulus", encodedModulus));
        if (!modulus.isPresent()) {
            return Optional.empty();
        }
        Optional exponent = key.getStringProperty("e").flatMap(encodedExponent -> JwksDecoder.decodeBigint(keyId, "exponent", encodedExponent));
        if (!exponent.isPresent()) {
            return Optional.empty();
        }
        return Optional.of(new JwkRsaPublicKey(keyId, (BigInteger)exponent.get(), (BigInteger)modulus.get()));
    }

    public static Optional<JwkEcPublicKey> tryDecodeEcKey(String keyId, JsonKey key) {
        Optional<String> curveName = key.getStringProperty("crv");
        Optional curve = curveName.flatMap(EcCurve::tryGet);
        if (!curve.isPresent()) {
            log.error("JWK EC %s curve '%s' is not supported", new Object[]{keyId, curveName});
            return Optional.empty();
        }
        Optional x = key.getStringProperty("x").flatMap(encodedX -> JwksDecoder.decodeBigint(keyId, "x", encodedX));
        if (!x.isPresent()) {
            return Optional.empty();
        }
        Optional y = key.getStringProperty("y").flatMap(encodedY -> JwksDecoder.decodeBigint(keyId, "y", encodedY));
        if (!y.isPresent()) {
            return Optional.empty();
        }
        ECPoint w = new ECPoint((BigInteger)x.get(), (BigInteger)y.get());
        return Optional.of(new JwkEcPublicKey(keyId, (ECParameterSpec)curve.get(), w));
    }

    private static Optional<BigInteger> decodeBigint(String keyId, String fieldName, String encodedNumber) {
        try {
            return Optional.of(new BigInteger(1, Base64.getUrlDecoder().decode(encodedNumber)));
        }
        catch (IllegalArgumentException e) {
            log.error((Throwable)e, "JWK %s %s is not a valid number", new Object[]{keyId, fieldName});
            return Optional.empty();
        }
    }

    public static class JsonKey {
        private final String kty;
        private final Optional<String> kid;
        private final Map<String, Object> other = new HashMap<String, Object>();

        @JsonCreator
        public JsonKey(@JsonProperty(value="kty") String kty, @JsonProperty(value="kid") Optional<String> kid) {
            this.kty = Objects.requireNonNull(kty, "kty is null");
            this.kid = Objects.requireNonNull(kid, "kid is null");
        }

        public String getKty() {
            return this.kty;
        }

        public Optional<String> getKid() {
            return this.kid;
        }

        public Optional<String> getStringProperty(String name) {
            Object value = this.other.get(name);
            if (value instanceof String && !((String)value).isEmpty()) {
                return Optional.of((String)value);
            }
            return Optional.empty();
        }

        @JsonAnySetter
        public void set(String name, Object value) {
            this.other.put(name, value);
        }
    }

    public static class JsonKeys {
        private final List<JsonKey> keys;

        @JsonCreator
        public JsonKeys(@JsonProperty(value="keys") List<JsonKey> keys) {
            this.keys = ImmutableList.copyOf((Collection)Objects.requireNonNull(keys, "keys is null"));
        }

        public List<JsonKey> getKeys() {
            return this.keys;
        }
    }
}

