package in.sourceshift.genericmodules.securityutils.web.jwt;

import java.io.IOException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;

import in.sourceshift.genericmodules.securityutils.BaseEncoder;
import in.sourceshift.genericmodules.securityutils.exception.SecurityUtilsException;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public class PayloadProcessor {
    private SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyyMMddHHmmss");

    ObjectMapper mappernew = new ObjectMapper();
    ObjectNode keyvalue = mappernew.createObjectNode();

    private final Set<String> predefinepayload = new HashSet<>(Arrays.asList("iss", "sub", "aud", "exp", "nbf", "iat", "jti"));

    public PayloadProcessor() throws SecurityUtilsException {

    }

    public PayloadProcessor setProperty(String key, String value) {
        if (predefinepayload.contains(key)) {
            log.warn("Use inbuild method instade of raw headers");
        }
        keyvalue.put(key, value);
        return this;

    }

    public String getProperty(String key) {
        if (predefinepayload.contains(key)) {
            log.warn("Use inbuild method instade of raw headers");
        }
        return keyvalue.get(key).asText();

    }

    public PayloadProcessor setIssuer(String issuer) {
        keyvalue.put("iss", issuer);
        return this;
    }

    public PayloadProcessor setSubject(String subject) {
        keyvalue.put("sub", subject);
        return this;
    }

    public PayloadProcessor setAudience(String audience) {
        keyvalue.put("aud", audience);
        return this;
    }

    public PayloadProcessor setExpirationTime(Date expirationTime) throws SecurityUtilsException {
        if (getIssuedat() == null) {
            setIssuedat(new Date());
        }
        keyvalue.put("exp", simpleDateFormat.format(expirationTime));
        return this;
    }

    public PayloadProcessor setExpirationTime(int validitycalunit, int validity) throws SecurityUtilsException {
        if (getIssuedat() == null) {
            setIssuedat(new Date());
        }
        Calendar cal = Calendar.getInstance();
        cal.setTime(getIssuedat());
        cal.add(validitycalunit, validity);
        setExpirationTime(cal.getTime());
        return this;
    }

    public PayloadProcessor setNotBefore(Date notBefore) throws SecurityUtilsException {
        if (getIssuedat() == null) {
            setIssuedat(new Date());
        }
        keyvalue.put("nbf", simpleDateFormat.format(notBefore));
        return this;
    }

    public PayloadProcessor setNotBefore(int validitycalunit, int validity) throws SecurityUtilsException {
        if (getIssuedat() == null) {
            setIssuedat(new Date());
        }
        Calendar cal = Calendar.getInstance();
        cal.setTime(getIssuedat());
        cal.add(validitycalunit, validity);
        setNotBefore(cal.getTime());
        return this;
    }

    public PayloadProcessor setIssuedat(Date issuedat) {
        keyvalue.put("iat", simpleDateFormat.format(issuedat));
        return this;
    }

    public PayloadProcessor setJWTID(String jWTID) {
        keyvalue.put("jti", jWTID);
        return this;
    }

    public String getIssuer() {
        return keyvalue.get("iss").asText();
    }

    public String getSubject() {
        return keyvalue.get("sub").asText();
    }

    public String getAudience() {
        return keyvalue.get("aud").asText();
    }

    public Date getExpirationTime() throws SecurityUtilsException {
        try {
            if ((keyvalue.path("exp").isMissingNode())) {
                return null;
            }
            return simpleDateFormat.parse(keyvalue.get("exp").asText());
        } catch (ParseException e) {
            throw new SecurityUtilsException("Unable to parse ExpirationTime form JWT Payload", e);
        }
    }

    public Date getNotBefore() throws SecurityUtilsException {
        try {
            if ((keyvalue.path("nbf").isMissingNode())) {
                return null;
            }
            return simpleDateFormat.parse(keyvalue.get("nbf").asText());
        } catch (ParseException e) {
            throw new SecurityUtilsException("Unable to parse NotBefore form JWT Payload", e);
        }
    }

    public Date getIssuedat() throws SecurityUtilsException {
        try {
            if ((keyvalue.path("iat").isMissingNode())) {
                return null;
            }
            return simpleDateFormat.parse(keyvalue.get("iat").asText());
        } catch (ParseException e) {
            throw new SecurityUtilsException("Unable to parse Issuedat form JWT Payload", e);
        }
    }

    public String getJWTID() {
        return keyvalue.get("jti").asText();
    }

    public String getbase64urlencodedJson() throws SecurityUtilsException {

        String json = keyvalue.toString();
        String base64json = BaseEncoder.encodeBase64toString(json);
        String b64url = URLEncoder.encode(base64json, StandardCharsets.UTF_8);
        return b64url;

    }

    public PayloadProcessor decodefrombase64urlencodedJson(String base64urlencodedJson) throws SecurityUtilsException {
        String base64json;
        base64json = URLDecoder.decode(base64urlencodedJson, StandardCharsets.UTF_8);
        String json = BaseEncoder.decodeBase64toString(base64json);
        try {
            keyvalue = (ObjectNode) mappernew.readTree(json);
        } catch (IOException e) {
            throw new SecurityUtilsException("Unable to decode JWTPayload Json", e);
        }
        return this;

    }

    public PayloadProcessor decodefromJson(String payloadJson) throws SecurityUtilsException {
        try {
            keyvalue = (ObjectNode) mappernew.readTree(payloadJson);
        } catch (IOException e) {
            throw new SecurityUtilsException("Unable to decode JWTPayload Json", e);
        }
        return this;

    }

    @Override
    public String toString() {
        return getPayloadJson();
    }

    public String getPayloadJson() {
        return keyvalue.toString();
    }
}
