package in.sourceshift.genericmodules.securityutils.web.jwt;

import java.net.URLDecoder;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.Date;

import in.sourceshift.genericmodules.securityutils.HashwithSecret;
import in.sourceshift.genericmodules.securityutils.exception.SecurityUtilsException;
import in.sourceshift.genericmodules.securityutils.hashalgorithms.HashType;

public class JWT implements JWTBuilder, JWTDecoder {

    private JWT() {

    }

    private JWT(final String token) throws SecurityUtilsException {
        this.token = token;

        String[] jwtsplit = token.split("\\.");
        base64urlencodedJsonHeader = jwtsplit[0];
        base64urlencodedJsonpayload = jwtsplit[1];
        signaturebase64url = jwtsplit[2];
        headerProcessor = new HeaderProcessor().decodefrombase64urlencodedJson(base64urlencodedJsonHeader);
        payloadProcessor = new PayloadProcessor().decodefrombase64urlencodedJson(base64urlencodedJsonpayload);
        headerJson = headerProcessor.getHeaderJson();
        payloadJson = payloadProcessor.getPayloadJson();
    }

    @Override
    public HeaderProcessor getHeaderProcessor() {
        return headerProcessor;
    }

    @Override
    public PayloadProcessor getPayloadProcessor() {
        return payloadProcessor;
    }

    private HeaderProcessor headerProcessor;
    private PayloadProcessor payloadProcessor;
    private String headerJson;
    private String base64urlencodedJsonHeader;
    private String payloadJson;
    private String base64urlencodedJsonpayload;
    private String signaturebase64url;
    private String secret;
    private String token;

    public static JWTBuilder getBuilder() {
        return new JWT();
    }

    public static JWTDecoder getDecoder(final String token) throws SecurityUtilsException {
        return new JWT(token);
    }

    private void build() throws SecurityUtilsException {

        String headandpay = base64urlencodedJsonHeader + "." + base64urlencodedJsonpayload;
        String tempsign;
        if (headerProcessor.getAlgorithm().equals(JWTHashType.HS256)) {
            tempsign = HashwithSecret.setData(headandpay).setSecret(secret).setHashtype(HashType.HMAC_SHA256).create();
        } else {
            throw new SecurityUtilsException("HashType not supported");
        }

        signaturebase64url = URLEncoder.encode(tempsign, StandardCharsets.UTF_8);
        token = headandpay + "." + signaturebase64url;

    }

    @Override
    public boolean validate() throws SecurityUtilsException {

        return verifyHash() && verifyexptime();
    }

    private boolean verifyHash() throws SecurityUtilsException {
        HashType hmac;
        String oldhash;
        String headerandpayloadtoken = base64urlencodedJsonHeader + "." + base64urlencodedJsonpayload;
        oldhash = URLDecoder.decode(signaturebase64url, StandardCharsets.UTF_8);
        if (headerProcessor.getAlgorithm().equals(JWTHashType.HS256)) {
            hmac = HashType.HMAC_SHA256;
        } else {
            throw new SecurityUtilsException("HashType not supported");
        }

        if (!HashwithSecret.setData(headerandpayloadtoken).setHashtype(hmac).setSecret(secret).verify(oldhash)) {
            throw new SecurityUtilsException("Regenerated Hash integrity varification failed");
        }

        return true;
    }

    private boolean verifyexptime() throws SecurityUtilsException {
        Date Current = new Date();

        if ((payloadProcessor.getIssuedat() != null) && (payloadProcessor.getNotBefore() != null)) {
            if ((payloadProcessor.getIssuedat().after(Current)) || (payloadProcessor.getIssuedat().after(payloadProcessor.getNotBefore()))) {
                throw new SecurityUtilsException("JWT integrity validation failed, JWT ID: " + payloadProcessor.getJWTID());
            }
        }

        if (payloadProcessor.getExpirationTime() != null) {
            if (payloadProcessor.getExpirationTime().before(Current)) {
                throw new SecurityUtilsException("JWT expired, JWT ID: " + payloadProcessor.getJWTID());
            }
        }

        if (payloadProcessor.getNotBefore() != null) {
            if (payloadProcessor.getNotBefore().after(Current)) {
                throw new SecurityUtilsException("JWT is not yet available for use, JWT ID: " + payloadProcessor.getJWTID());
            }
        }

        return true;
    }

    @Override
    public String quickRebuild(String key, String value) throws SecurityUtilsException {
        payloadProcessor.setProperty(key, value);
        JWTBuilder jwt = JWT.getBuilder().setHeaderJson(getHeaderJson()).setPayloadJson(payloadProcessor.getPayloadJson());
        jwt.setSecret(secret);
        return jwt.getToken();
    }

    @Override
    public String getHeaderJson() {
        return headerJson;
    }

    @Override
    public String getPayloadJson() {
        return payloadJson;
    }

    @Override
    public JWTBuilder setHeaderJson(String headerJson) throws SecurityUtilsException {
        headerProcessor = new HeaderProcessor().decodefromJson(headerJson);
        base64urlencodedJsonHeader = headerProcessor.getbase64urlencodedJson();
        this.headerJson = headerJson;
        return this;
    }

    @Override
    public JWTBuilder setPayloadJson(String payloadJson) throws SecurityUtilsException {
        payloadProcessor = new PayloadProcessor().decodefromJson(payloadJson);
        base64urlencodedJsonpayload = payloadProcessor.getbase64urlencodedJson();
        this.payloadJson = payloadJson;
        return this;
    }

    @Override
    public void setSecret(String secret) {
        this.secret = secret;
    }

    @Override
    public String getToken() throws SecurityUtilsException {
        build();
        return token;
    }

}
