package in.sourceshift.genericmodules.securityutils.otp;

import java.security.Key;
import java.time.Duration;
import java.time.Instant;

import in.sourceshift.genericmodules.securityutils.exception.SecurityUtilsException;

/**
 * <p>
 * Generates time-based one-time passwords (TOTP) as specified in <a href="https://tools.ietf.org/html/rfc6238">RFC&nbsp;6238</a>.
 * </p>
 * <p>
 * {@code TimeBasedOneTimePasswordGenerator} instances are thread-safe and may be shared between threads.
 * </p>
 */
public class TimeBasedOneTimePasswordGenerator extends HmacOneTimePasswordGenerator {
    private final Duration timeStep;

    /**
     * The default time-step for a time-based one-time password generator (30 seconds).
     */
    public static final Duration DEFAULT_TIME_STEP = Duration.ofSeconds(30);

    /**
     * A string identifier for the HMAC-SHA1 algorithm (required by HOTP and allowed by TOTP). HMAC-SHA1 is the default algorithm for TOTP.
     */
    public static final String TOTP_ALGORITHM_HMAC_SHA1 = "HmacSHA1";

    /**
     * A string identifier for the HMAC-SHA256 algorithm (allowed by TOTP).
     */
    public static final String TOTP_ALGORITHM_HMAC_SHA256 = "HmacSHA256";

    /**
     * A string identifier for the HMAC-SHA512 algorithm (allowed by TOTP).
     */
    public static final String TOTP_ALGORITHM_HMAC_SHA512 = "HmacSHA512";

    public TimeBasedOneTimePasswordGenerator() throws SecurityUtilsException {
        this(DEFAULT_TIME_STEP);
    }

    public TimeBasedOneTimePasswordGenerator(final Duration timeStep) throws SecurityUtilsException {
        this(timeStep, DEFAULT_PASSWORD_LENGTH);
    }

    public TimeBasedOneTimePasswordGenerator(final Duration timeStep, final int passwordLength) throws SecurityUtilsException {
        this(timeStep, passwordLength, TOTP_ALGORITHM_HMAC_SHA1);
    }

    public TimeBasedOneTimePasswordGenerator(final Duration timeStep, final int passwordLength, final String algorithm) throws SecurityUtilsException {
        super(passwordLength, algorithm);

        this.timeStep = timeStep;
    }

    public int generateOneTimePassword(final Key key, final Instant timestamp) throws SecurityUtilsException {
        return this.generateOneTimePassword(key, timestamp.toEpochMilli() / timeStep.toMillis());
    }

    public Duration getTimeStep() {
        return timeStep;
    }
}
