package in.sourceshift.genericmodules.securityutils.otp;

import java.security.InvalidKeyException;
import java.security.Key;
import java.security.NoSuchAlgorithmException;

import javax.crypto.Mac;
import javax.crypto.ShortBufferException;

import in.sourceshift.genericmodules.securityutils.exception.SecurityUtilsException;

/**
 * <p>
 * Generates HMAC-based one-time passwords (HOTP) as specified in <a href="https://tools.ietf.org/html/rfc4226">RFC&nbsp;4226</a>.
 * </p>
 * <p>
 * {@code HmacOneTimePasswordGenerator} instances are thread-safe and may be shared between threads.
 * </p>
 */
public class HmacOneTimePasswordGenerator {
    private final Mac mac;
    private final int passwordLength;

    private final byte[] buffer;
    private final int modDivisor;

    /**
     * The default length, in decimal digits, for one-time passwords.
     */
    public static final int DEFAULT_PASSWORD_LENGTH = 6;

    /**
     * The HMAC algorithm specified by the HOTP standard.
     */
    public static final String HOTP_HMAC_ALGORITHM = "HmacSHA1";

    public HmacOneTimePasswordGenerator() throws SecurityUtilsException {
        this(DEFAULT_PASSWORD_LENGTH);
    }

    public HmacOneTimePasswordGenerator(final int passwordLength) throws SecurityUtilsException {
        this(passwordLength, HOTP_HMAC_ALGORITHM);
    }

    public HmacOneTimePasswordGenerator(final int passwordLength, final String algorithm) throws SecurityUtilsException {
        try {
            mac = Mac.getInstance(algorithm);
        } catch (NoSuchAlgorithmException e) {
            throw new SecurityUtilsException(e);
        }

        switch (passwordLength) {
        case 6: {
            modDivisor = 1_000_000;
            break;
        }

        case 7: {
            modDivisor = 10_000_000;
            break;
        }

        case 8: {
            modDivisor = 100_000_000;
            break;
        }

        default: {
            throw new SecurityUtilsException("Password length must be between 6 and 8 digits.");
        }
        }

        this.passwordLength = passwordLength;
        buffer = new byte[mac.getMacLength()];
    }

    public synchronized int generateOneTimePassword(final Key key, final long counter) throws SecurityUtilsException {
        try {
            mac.init(key);
        } catch (InvalidKeyException e1) {
            throw new SecurityUtilsException(e1);
        }

        buffer[0] = (byte) ((counter & 0xff00000000000000L) >>> 56);
        buffer[1] = (byte) ((counter & 0x00ff000000000000L) >>> 48);
        buffer[2] = (byte) ((counter & 0x0000ff0000000000L) >>> 40);
        buffer[3] = (byte) ((counter & 0x000000ff00000000L) >>> 32);
        buffer[4] = (byte) ((counter & 0x00000000ff000000L) >>> 24);
        buffer[5] = (byte) ((counter & 0x0000000000ff0000L) >>> 16);
        buffer[6] = (byte) ((counter & 0x000000000000ff00L) >>> 8);
        buffer[7] = (byte) (counter & 0x00000000000000ffL);

        mac.update(buffer, 0, 8);

        try {
            mac.doFinal(buffer, 0);
        } catch (final ShortBufferException e) {
            // We allocated the buffer to (at least) match the size of the MAC length at
            // construction time, so this
            // should never happen.
            throw new RuntimeException(e);
        }

        final int offset = buffer[buffer.length - 1] & 0x0f;

        return (((buffer[offset] & 0x7f) << 24) | ((buffer[offset + 1] & 0xff) << 16) | ((buffer[offset + 2] & 0xff) << 8) | (buffer[offset + 3] & 0xff)) % modDivisor;
    }

    public int getPasswordLength() {
        return passwordLength;
    }

    public String getAlgorithm() {
        return mac.getAlgorithm();
    }
}
