package im.qingtui.qbee.open.platform.third.token.common.utils;

import static im.qingtui.qbee.open.platfrom.base.common.constants.TokenConstants.TOKEN_PREFIX;

import com.alibaba.fastjson.JSONObject;
import com.auth0.jwt.JWT;
import com.auth0.jwt.JWTVerifier;
import com.auth0.jwt.algorithms.Algorithm;
import com.auth0.jwt.exceptions.JWTDecodeException;
import com.auth0.jwt.interfaces.DecodedJWT;
import im.qingtui.qbee.open.platform.third.token.common.constants.SecurityConstants;
import im.qingtui.qbee.open.platform.third.token.common.exception.QbeeSSOException;
import im.qingtui.qbee.open.platform.third.token.model.dto.QBeeTokenExpDTO;
import im.qingtui.qbee.open.platform.third.token.common.constants.ExceptionConstants;
import im.qingtui.qbee.open.platfrom.base.common.utils.Base64Utils;
import im.qingtui.qbee.open.platfrom.base.common.utils.ConfigUtils;
import java.util.Date;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.StringUtils;

@Slf4j
public class JwtUtils {

    /**
     * token包含数据部分长度（用.隔开）
     */
    private final static int TOKEN_PARTY_LENGTH = 4;

    /**
     * 一分钟的毫秒数
     */
    private final static long ONE_MINUTES_MILLIS = 60 * 1000L;

    /**
     * 一秒的毫秒数
     */
    private final static long ONE_SECOND_MILLIS = 1000L;

    /**
     * @author jessie 校验token是否正确 异常会直接抛错
     */
    public static void verify(String token) {
        try {
            String finalToken = dealToken(token);
            String secret = getClaim(finalToken, SecurityConstants.EMPLOYEE_ID) + ConfigUtils.getValue(SecurityConstants.TOKEN_SECRET);
            Algorithm algorithm = Algorithm.HMAC256(secret);
            JWTVerifier verifier = JWT.require(algorithm).build();
            verifier.verify(finalToken);
        } catch (Exception e) {
            log.error("token验证失败", e);
            throw new QbeeSSOException(ExceptionConstants.TOKEN_INVALID);
        }
    }

    /**
     * 处理整理token，去掉Bear和最后一截数据
     * @param token 请求的第三方token
     * @return 整理后的token
     */
    public static String dealToken(String token) {
        if (StringUtils.isEmpty(token)) {
            throw new QbeeSSOException(ExceptionConstants.TOKEN_INVALID);
        }
        // 校验token
        String finalToken = token.startsWith(TOKEN_PREFIX) ? token.substring(TOKEN_PREFIX.length()) : token;
        if (finalToken.split(SecurityConstants.TOKEN_SEPARATION_RPX).length == TOKEN_PARTY_LENGTH ) {
            finalToken =  finalToken.substring(0, finalToken.lastIndexOf(SecurityConstants.TOKEN_SEPARATION));
        }
        return finalToken;
    }

    /**
     * 获得Token中的信息无需secret解密也能获得
     */
    public static String getClaim(String token, String claim) {
        try {
            DecodedJWT jwt = JWT.decode(token);
            return jwt.getClaim(claim).asString();
        } catch (JWTDecodeException e) {
            log.error("获取token参数失败：", e);
            return null;
        }
    }

    /**
     * 通过员工id、用户id， 当前时间和过期时间获取token签名
     * @param employeeId 当前签名请求用户的员工id
     * @param userId 当前签名请求用户的账号id
     * @param currentTimeMillis 当前时间
     * @param expireTime 过期时间
     * @return 签名内容
     */
    public static String sign(String employeeId, String userId, Long currentTimeMillis, Date expireTime) {
        // 帐号加JWT私钥加密
        String secret = employeeId + ConfigUtils.getValue(SecurityConstants.TOKEN_SECRET);
        // 此处过期时间，单位：毫秒
        Algorithm algorithm = Algorithm.HMAC256(secret);
        return JWT.create()
            .withClaim(SecurityConstants.EMPLOYEE_ID, employeeId)
            .withClaim(SecurityConstants.USER_ID, userId)
            .withClaim(SecurityConstants.CURRENT_TIME_MILLIS, currentTimeMillis)
            .withExpiresAt(expireTime)
            .sign(algorithm);
    }

    /**
     * 通过员工id、用户id， 轻蜂的token生成三方token签名
     * @param employeeId 当前签名请求用户的员工id
     * @param qBeeToken 轻蜂token
     * @return 签名内容
     */
    public static String signByQBeeToken(String employeeId, String userId, String qBeeToken) {
        String qBeeTokenClaim = qBeeToken.split(SecurityConstants.TOKEN_SEPARATION_RPX)[1];
        QBeeTokenExpDTO qBeeTokenExpDTO = JSONObject.parseObject(Base64Utils.decode(qBeeTokenClaim),
            QBeeTokenExpDTO.class);
        long currentTimeMillis = System.currentTimeMillis();
        String configExpireTime = ConfigUtils.getValue(SecurityConstants.EXPIRE_TIME);
        long qBeeTokenExpireTime = qBeeTokenExpDTO.getExp() * ONE_SECOND_MILLIS;
        long thirdTokenExpireTime = qBeeTokenExpireTime;
        if (!StringUtils.isEmpty(configExpireTime)
            && Base64Utils.isInteger(configExpireTime)
            && qBeeTokenExpireTime >= (currentTimeMillis + Long.parseLong(configExpireTime) * ONE_MINUTES_MILLIS)) {
            thirdTokenExpireTime = currentTimeMillis + Long.parseLong(configExpireTime) * ONE_MINUTES_MILLIS;
        }
        Date expireTime = new Date(thirdTokenExpireTime);
        return sign(employeeId, userId, currentTimeMillis, expireTime);
    }
}
