package im.qingtui.qbee.open.platfrom.portal.service;

import static im.qingtui.qbee.open.platfrom.portal.common.constants.UrlConstants.*;

import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import im.qingtui.qbee.open.platfrom.base.model.enums.HttpRequestTypeEnum;
import im.qingtui.qbee.open.platfrom.base.model.param.base.BaseCodeParam;
import im.qingtui.qbee.open.platfrom.base.model.vo.base.BaseHttpVO;
import im.qingtui.qbee.open.platfrom.portal.model.param.dimension.CreateDimensionParam;
import im.qingtui.qbee.open.platfrom.portal.model.param.dimension.EditDimensionParam;
import im.qingtui.qbee.open.platfrom.base.model.vo.base.BaseId;
import im.qingtui.qbee.open.platfrom.base.model.vo.base.BaseList;
import im.qingtui.qbee.open.platfrom.base.model.vo.base.BaseResult;
import im.qingtui.qbee.open.platfrom.base.model.vo.base.BaseData;
import im.qingtui.qbee.open.platfrom.portal.model.vo.dimension.DimensionInfo;
import im.qingtui.qbee.open.platfrom.portal.model.vo.dimension.DimensionTree;
import im.qingtui.qbee.open.platfrom.base.common.exception.IllegalRequestException;
import im.qingtui.qbee.open.platfrom.base.common.utils.HttpClientRequestUtils;
import im.qingtui.qbee.open.platfrom.base.common.utils.TokenUtils;

/**
 * 维度相关接口
 *
 * @author shizhen
 */
public class DimensionService {


    /**
     * 获取服务号所属团队的维度列表（不带token,取配置文件的服务号信息获取token）
     *
     * @return 服务号所属团队的维度列表
     */
    public static BaseList<DimensionInfo> getDimensionList() {
        return getDimensionList(TokenUtils.getToken());
    }

    /**
     * 获取服务号所属团队的维度列表
     *
     * @param token 开放平台token
     * @return 服务号所属团队的维度列表
     */
    public static BaseList<DimensionInfo> getDimensionList(String token) {
        BaseHttpVO baseHttpVO = HttpClientRequestUtils.getBaseHttpVO(HttpRequestTypeEnum.GET, GET_DIMENSION_LIST_URL, token);
        if (0 == baseHttpVO.getCode()) {
            BaseList<DimensionInfo> dimensionListVO = JSONObject.parseObject(baseHttpVO.getDataJson().toString(), new TypeReference<BaseList<DimensionInfo>>(){});
            return dimensionListVO;
        } else {
            throw new IllegalRequestException(new BaseResult(baseHttpVO));
        }
    }

    /**
     * 创建维度（不带token,取配置文件的服务号信息获取token）
     *
     * @param createDimensionParam 添加维度参数
     * @return 创建的维度id
     */
    public static BaseData<BaseId> createDimension(CreateDimensionParam createDimensionParam) {
        return createDimension(createDimensionParam, TokenUtils.getToken());
    }

    /**
     * 创建维度
     *
     * @param createDimensionParam 添加维度参数
     * @param token 开放平台token
     * @return 创建的维度id
     */
    public static BaseData<BaseId> createDimension(CreateDimensionParam createDimensionParam, String token) {
        BaseHttpVO baseHttpVO = HttpClientRequestUtils.getBaseHttpVO(HttpRequestTypeEnum.POST, CREAT_DIMENSION_URL, token, createDimensionParam);
        if (0 == baseHttpVO.getCode()) {
            return new BaseData<>(JSONObject.parseObject(baseHttpVO.getDataJson().toString(), BaseId.class));
        } else {
            throw new IllegalRequestException(new BaseResult(baseHttpVO));
        }
    }

    /**
     * 编辑维度（不带token,取配置文件的服务号信息获取token）
     *
     * @param editDimensionParam 编辑维度参数
     */
    public static void editDimension(EditDimensionParam editDimensionParam) {
        editDimension(editDimensionParam, TokenUtils.getToken());
    }

    /**
     * 编辑维度
     *
     * @param editDimensionParam 编辑维度参数
     * @param token 开放平台token
     */
    public static void editDimension(EditDimensionParam editDimensionParam, String token) {
        BaseHttpVO baseHttpVO = HttpClientRequestUtils.getBaseHttpVO(HttpRequestTypeEnum.PUT, EDIT_DIMENSION_URL, token, editDimensionParam);
        if (0 != baseHttpVO.getCode()) {
            throw new IllegalRequestException(new BaseResult(baseHttpVO));
        }
    }

    /**
     * 删除维度（不带token,取配置文件的服务号信息获取token）
     *
     * @param id 维度id
     */
    public static void deleteDimension(String id) {
        deleteDimension(id, TokenUtils.getToken());
    }

    /**
     * 删除维度
     *
     * @param id 维度id
     * @param token 开放平台token
     */
    public static void deleteDimension(String id, String token) {
        BaseHttpVO baseHttpVO = HttpClientRequestUtils.getBaseHttpVO(HttpRequestTypeEnum.DELETE, DELETE_DIMENSION_URL, token, new BaseId(id));
        if (0 != baseHttpVO.getCode()) {
            throw new IllegalRequestException(new BaseResult(baseHttpVO));
        }
    }

    /**
     * 根据维度id获取维度信息（不带token,取配置文件的服务号信息获取token）
     *
     * @param id 维度id
     * @return 查询的维度信息
     */
    public static BaseData<DimensionInfo> getDimensionInfoById(String id) {
        return getDimensionInfoById(id, TokenUtils.getToken());
    }

    /**
     * 根据维度id获取维度信息
     *
     * @param id 维度id
     * @param token 开放平台token
     * @return 查询的维度信息
     */
    public static BaseData<DimensionInfo> getDimensionInfoById(String id, String token) {
        BaseHttpVO baseHttpVO = HttpClientRequestUtils.getBaseHttpVO(HttpRequestTypeEnum.GET, GET_DIMENSION_BY_ID_URL, token, new BaseId(id));
        if (0 == baseHttpVO.getCode()) {
            return new BaseData<>(JSONObject.parseObject(baseHttpVO.getDataJson().toString(), DimensionInfo.class));
        } else {
            throw new IllegalRequestException(new BaseResult(baseHttpVO));
        }
    }


    /**
     * 根据维度编码获取维度信息（不带token,取配置文件的服务号信息获取token）
     *
     * @param code 维度编码
     * @return 查询的维度信息
     */
    public static BaseData<DimensionInfo> getDimensionInfoByCode(String code) {
        return getDimensionInfoByCode(code, TokenUtils.getToken());
    }

    /**
     * 根据维度编码获取维度信息
     *
     * @param code 维度编码
     * @param token 开放平台token
     * @return 查询的维度信息
     */
    public static BaseData<DimensionInfo> getDimensionInfoByCode(String code, String token) {
        BaseHttpVO baseHttpVO = HttpClientRequestUtils.getBaseHttpVO(HttpRequestTypeEnum.GET, GET_DIMENSION_BY_CODE_URL, token, new BaseCodeParam(code));
        if (0 == baseHttpVO.getCode()) {
            return new BaseData<>(JSONObject.parseObject(baseHttpVO.getDataJson().toString(), DimensionInfo.class));
        } else {
            throw new IllegalRequestException(new BaseResult(baseHttpVO));
        }
    }

    /**
     * 根据维度id获取该维度下的维度树（不带token,取配置文件的服务号信息获取token）
     *
     * @param id 维度id
     * @return 查询维度下的维度树
     */
    public static BaseList<DimensionTree> getDimensionTree(String id) {
        return getDimensionTree(id, TokenUtils.getToken());
    }

    /**
     * 根据维度id获取维度下的维度树
     *
     * @param id 维度id
     * @param token 开放平台token
     * @return 查询维度下的维度树
     */
    public static BaseList<DimensionTree> getDimensionTree(String id, String token) {
        BaseHttpVO baseHttpVO = HttpClientRequestUtils.getBaseHttpVO(HttpRequestTypeEnum.GET, GET_DIMENSION_TREE_BY_ID_URL, token, new BaseId(id));
        if (0 == baseHttpVO.getCode()) {
            return JSONObject.parseObject(baseHttpVO.getDataJson().toString(), new TypeReference<BaseList<DimensionTree>>(){});
        } else {
            throw new IllegalRequestException(new BaseResult(baseHttpVO));
        }
    }
}