package im.qingtui.qbee.open.platfrom.schedule.service;

import static im.qingtui.qbee.open.platfrom.base.common.utils.DateUtils.TWELVE_HOURS_MILLIS;
import static im.qingtui.qbee.open.platfrom.schedule.model.constant.UrlConstant.GROUP_PAGE_URL;
import static im.qingtui.qbee.open.platfrom.schedule.model.constant.UrlConstant.GROUP_TURN_DATE_REGION_URL;
import static im.qingtui.qbee.open.platfrom.schedule.model.constant.UrlConstant.GROUP_TURN_DATE_URL;
import static im.qingtui.qbee.open.platfrom.schedule.model.constant.UrlConstant.GROUP_URL;
import static im.qingtui.qbee.open.platfrom.schedule.model.constant.UrlConstant.HOLIDAY_URL;

import com.alibaba.fastjson.TypeReference;
import im.qingtui.qbee.open.platfrom.base.common.utils.CollectionUtils;
import im.qingtui.qbee.open.platfrom.base.common.utils.DateUtils;
import im.qingtui.qbee.open.platfrom.base.common.utils.HttpClientRequestUtils;
import im.qingtui.qbee.open.platfrom.base.common.utils.TokenUtils;
import im.qingtui.qbee.open.platfrom.base.common.utils.UrlUtil;
import im.qingtui.qbee.open.platfrom.base.model.param.base.IdListParam;
import im.qingtui.qbee.open.platfrom.base.model.param.base.PageParam;
import im.qingtui.qbee.open.platfrom.base.model.vo.base.BaseHttpVO;
import im.qingtui.qbee.open.platfrom.base.model.vo.base.BaseList;
import im.qingtui.qbee.open.platfrom.base.model.vo.base.BasePage;
import im.qingtui.qbee.open.platfrom.schedule.model.param.GroupTurnDateParam;
import im.qingtui.qbee.open.platfrom.schedule.model.param.GroupTurnDateRegionParam;
import im.qingtui.qbee.open.platfrom.schedule.model.vo.WorkGroupInfo;
import im.qingtui.qbee.open.platfrom.schedule.model.vo.WorkGroupTurnVO;
import im.qingtui.qbee.open.platfrom.schedule.model.vo.WorkHolidayInfo;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import lombok.NonNull;
import lombok.val;

/**
 * <p> title: 班组管理
 * <p> time: 2021/4/25 4:18 下午
 * <p> Copyright © 2010-2021 CISDI Info. All Rights Reserved.
 *
 * @author 阳君
 * @version 1.0.0
 */
public class WorkGroupService {

    /**
     * 通过班组 id 获取班组详情
     *
     * @param groupId 班组id
     * @return 班组信息
     */
    public static WorkGroupInfo getByGroupId(String groupId) {
        val list = listByGroupId(Collections.singletonList(groupId));
        return CollectionUtils.get(list, 0);
    }

    /**
     * 通过班组id批量获取班组详情
     *
     * @param groupIdList 班组id
     * @return List WorkTurnVO
     */
    public static List<WorkGroupInfo> listByGroupId(List<String> groupIdList) {
        val baseHttpVO = HttpClientRequestUtils.getRequest(UrlUtil.getApiUrl(GROUP_URL), TokenUtils.getToken(), new IdListParam(groupIdList));
        return HttpClientRequestUtils.convertToList(baseHttpVO, new TypeReference<BaseList<WorkGroupInfo>>() {
        });
    }

    /**
     * 获取排班组列表
     *
     * @param param 分页信息
     * @return 班组列表
     */
    public static BasePage<WorkGroupInfo> page(PageParam param) {
        BaseHttpVO baseHttpVO = HttpClientRequestUtils.getRequest(UrlUtil.getApiUrl(GROUP_PAGE_URL), TokenUtils.getToken(), param);
        return HttpClientRequestUtils.convertToData(baseHttpVO, new TypeReference<BasePage<WorkGroupInfo>>() {
        });
    }

    /**
     * 根据日期查询当天排班信息
     *
     * @param gmtDate 时间戳 毫秒级
     * @return list
     */
    public static List<WorkGroupTurnVO> getGroupTurnByDate(String gmtDate) {
        BaseHttpVO baseHttpVO = HttpClientRequestUtils.getRequest(UrlUtil.getApiUrl(GROUP_TURN_DATE_URL), TokenUtils.getToken(), new GroupTurnDateParam(gmtDate));
        return HttpClientRequestUtils.convertToList(baseHttpVO, new TypeReference<BaseList<WorkGroupTurnVO>>() {
        });
    }

    /**
     * 根据日期查询时间点的排班信息
     *
     * @param gmtDate 时间戳 毫秒级
     * @return list
     */
    public static List<WorkGroupTurnVO> listGroupTurnByTime(String gmtDate) {
        val gmtTime = Long.parseLong(gmtDate);
        val result = new ArrayList<WorkGroupTurnVO>();
        // 连上两天
        if (DateUtils.distanceStartTime(gmtTime) < TWELVE_HOURS_MILLIS) {
            val yesterdayList = getGroupTurnByDate(String.valueOf(gmtTime - TWELVE_HOURS_MILLIS));
            for (val item : yesterdayList) {
                if (DateUtils.betweenTimeHms2(gmtTime, item.getTurn().getGmtStart(), item.getTurn().getGmtEnd())) {
                    result.add(item);
                }
            }
        }
        // 今天
        val todayList = getGroupTurnByDate(gmtDate);
        for (val item : todayList) {
            if (DateUtils.betweenTimeHms(gmtTime, item.getTurn().getGmtStart(), item.getTurn().getGmtEnd())) {
                result.add(item);
            }
        }
        return result;
    }

    /**
     * 通过时间和编码正则表达式获取时间点的排班信息
     *
     * @param gmtDate   时间戳
     * @param codeRegex 编码正则表达式
     * @return 排版信息
     */
    public static List<WorkGroupTurnVO> listGroupTurnByTimeRegex(String gmtDate, @NonNull String codeRegex) {
        List<WorkGroupTurnVO> list = listGroupTurnByTime(gmtDate);
        return filterWorkGroupByCodeRegex(list, codeRegex);
    }

    /**
     * 按照编码正则表达式过滤班组信息
     *
     * @param list      需要过滤的数组
     * @param codeRegex 编码正则表达式
     * @return 排版信息
     */
    public static List<WorkGroupTurnVO> filterWorkGroupByCodeRegex(List<WorkGroupTurnVO> list, @NonNull String codeRegex) {
        return list.stream().filter(item ->
            Pattern.matches(codeRegex, item.getGroup().getCode())
        ).collect(Collectors.toList());
    }

    /**
     * 根据起止日期范围查询排班信息
     *
     * @param param 起止时间参数封装
     * @return list
     */
    public static List<WorkGroupTurnVO> listGroupTurnByDate(GroupTurnDateRegionParam param) {
        BaseHttpVO baseHttpVO = HttpClientRequestUtils.getRequest(UrlUtil.getApiUrl(GROUP_TURN_DATE_REGION_URL), TokenUtils.getToken(), param);
        return HttpClientRequestUtils.convertToList(baseHttpVO, new TypeReference<BaseList<WorkGroupTurnVO>>() {
        });
    }

    /**
     * 通过节假日id获取节假日详情
     *
     * @param holidayId 节假日id
     * @return 节假日
     */
    public static WorkHolidayInfo getHolidayByHolidayId(String holidayId) {
        val list = listHolidayByHolidayId(Collections.singletonList(holidayId));
        return CollectionUtils.get(list, 0);
    }

    /**
     * 通过节假日id批量获取节假日详情
     *
     * @param holidayIdList 节假日id集合
     * @return 节假日
     */
    public static List<WorkHolidayInfo> listHolidayByHolidayId(List<String> holidayIdList) {
        BaseHttpVO baseHttpVO = HttpClientRequestUtils.getRequest(UrlUtil.getApiUrl(HOLIDAY_URL), TokenUtils.getToken(), new IdListParam(holidayIdList));
        val data = HttpClientRequestUtils.convertToData(baseHttpVO, new TypeReference<BaseList<WorkHolidayInfo>>() {
        });
        return BaseList.getData(data);
    }

}
