package com.idenfy.idenfySdk.liveness

import android.app.Activity
import android.app.Application
import android.arch.lifecycle.AndroidViewModel
import android.arch.lifecycle.MutableLiveData
import android.content.Intent
import android.graphics.Bitmap
import android.os.AsyncTask
import android.support.v4.app.FragmentActivity
import android.util.Base64
import android.util.Log
import android.util.Pair
import com.facetec.zoom.sdk.*
import com.idenfy.idenfySdk.helpers.SingleLiveEvent
import com.idenfy.idenfySdk.liveness.presentation.*
import com.idenfy.idenfySdk.liveness.ui.IdenfyLivenessSettings
import com.idenfy.idenfySdk.liveness.utils.ConstsLiveliness
import com.idenfy.idenfyliveness.*
import java.io.ByteArrayOutputStream
import java.util.*

class LivenessesViewModel(application: Application) : AndroidViewModel(application) {

    private var livenessHelper: IdenfyLivenessHelper? = null


    init {
        livenessHelper = getLivenessHelper()
        setupSources()
    }


    fun setupSources() {
        if (livenessHelper == null)
            return

    }


    fun getLivenessHelper(): IdenfyLivenessHelper? {
        try {
            Class.forName(ConstsLiveliness.CONST_OPTIONAL_LIVELINESS_ENABLED)
            if (livenessHelper == null)
                livenessHelper = com.idenfy.idenfyliveness.IdenfyLivenessHelper()
            return livenessHelper
        } catch (t: Throwable) {
            return null
        }

    }

    var zoomSDKInicializationResponse = MutableLiveData<Result<Pair<Boolean, String>>>()
    var zoomSessionResponse = MutableLiveData<LivenessSessionResponseState>()
    var zoomErrorOccured = SingleLiveEvent<Boolean>()

    var zoomSessionSuccess = SingleLiveEvent<Boolean>()

    var inicializationCount = 0

    val INICIALIZATION_COUNT_MAX = 5

    var resultsRetryCount = 0

    var RESULTS_COUNT_MAX = 3

    var isLoggingEnabled = false
    var IDENFY_SDK_LIVENESS_LOG = "idenfySDKLiveness"

    var developerToken = ""


    fun handleRetryingZoomPresentation(status: ZoomVerificationStatus) {
        if (status == ZoomVerificationStatus.FAILED_DUE_TO_INTERNAL_ERROR || status == ZoomVerificationStatus.FAILED_DUE_TO_CAMERA_ERROR) {
            if (resultsRetryCount < RESULTS_COUNT_MAX) {
                zoomSessionResponse.value = LivenessSessionNotSuccessfulNeedsAutoRetry(false, null, true)
            } else {
                zoomErrorOccured.value = true
            }
            resultsRetryCount++
        } else {
            zoomSessionResponse.value = (LivenessSessionNotSuccessfulNeedsUserRetry(false, null, true))
        }
    }

    fun handleZoomResults(token: String, data: Intent?) {
        try {
            Class.forName(ConstsLiveliness.CONST_OPTIONAL_LIVELINESS_ENABLED)
        } catch (e: ClassNotFoundException) {
            return
        }
        if (data == null)
            return
        val result = data.getParcelableExtra<ZoomVerificationResult>(
                ZoomSDK.EXTRA_VERIFY_RESULTS)
        if (isLoggingEnabled) {
            Log.d(IDENFY_SDK_LIVENESS_LOG, "livenessResultStatus: ${result.status.name}")
            Log.d(IDENFY_SDK_LIVENESS_LOG, "livenessResultMessage: ${result.status}")
        }
        if (result.status != ZoomVerificationStatus.USER_PROCESSED_SUCCESSFULLY) {
            if (result.status == ZoomVerificationStatus.OFFLINE_SESSIONS_EXCEEDED) {
                zoomErrorOccured.value = true
                return
            }
            handleRetryingZoomPresentation(result.status)


            return
        }
        zoomSessionSuccess.value = true
        val faceMetrics = result.faceMetrics
        if (faceMetrics != null) {

            val zoomFacemapBytes = faceMetrics.zoomFacemap
            val zoomFacemap = Base64.encodeToString(zoomFacemapBytes, Base64.NO_WRAP)
            val auditTrail = faceMetrics.auditTrail
            val auditTrails = ArrayList<String>()
            for (bitmap in auditTrail) {
                auditTrails.add(getBase64(bitmap))
            }
            val livenessCheck = LivenessCheck(token, result.sessionId, zoomFacemap, auditTrails)
            zoomSessionResponse.value = LivenessSessionSuccessful(true, livenessCheck, false)

        }
    }

    private fun getBase64(type: Bitmap): String {
        val baos = ByteArrayOutputStream()
        type.compress(Bitmap.CompressFormat.JPEG, 90, baos) //bm is the bitmap object
        val byteArrayImage = baos.toByteArray()
        val encodedImage = Base64.encodeToString(byteArrayImage, Base64.DEFAULT)
        type.recycle()
        return encodedImage
    }

    fun preloadZoomResources(activity: Activity?) {
        try {
            Class.forName(ConstsLiveliness.CONST_OPTIONAL_LIVELINESS_ENABLED)
        } catch (e: ClassNotFoundException) {
            return
        }

        AsyncTask.execute {
            if (activity != null)
                ZoomSDK.preload(activity)
        }
    }

    private val sharedPrivateKey = "-----BEGIN PUBLIC KEY-----\n" +
            "MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAyzEyODb/p3inHAP5S4kp\n" +
            "BTcwOdM39eI7Fmw4e1ZFako+cBTbkoSdfyHQjguVr2LPHe/bhzFZ1E5X/hdoUHd+\n" +
            "rBQUtVu9mCAgDoBOgknDqej/5iXSR3o4UL7WdiIsr7pEWbFnTRSH8hcygBC1F4oh\n" +
            "lshUbrGczhfCpx19L9OP7e5/jisTltVWYN4JHY/Jk4lw9qes+B7ds8DSny4xKHv3\n" +
            "B0Ebc9SXoYB1b/ibfrlYYKtZGFhbuLEMxuNuzt9o/cj6INykiNVXkI9xBGJIRor3\n" +
            "N8s4z8wdAkU/TBzwmKh1NC0sWDPynt6YfU5Gtdi5RSFnBdD9joDhCSCYEZ4eqXXz\n" +
            "rwIDAQAB\n" +
            "-----END PUBLIC KEY-----\n"

    fun initialiazeZoomSdk(activity: Activity?, primaryColor: Int, accentColor: Int, idenfy_logo: Int, developerToken: String,
                           idenfyLivenessUISettings: IdenfyLivenessUISettings?,
                           idenfyLivenessSettings: IdenfyLivenessSettings,
                           isloggingEnabled: Boolean?,
                           idenfyLivenessFeatureLogging: String?) {

        try {
            Class.forName(ConstsLiveliness.CONST_OPTIONAL_LIVELINESS_ENABLED)
        } catch (e: ClassNotFoundException) {
            return
        }


        if (isloggingEnabled != null) {
            this.isLoggingEnabled = isloggingEnabled
        }
        if (idenfyLivenessFeatureLogging != null) {
            this.IDENFY_SDK_LIVENESS_LOG = idenfyLivenessFeatureLogging
        }
        setCoreZoomSettings()
        this.developerToken = developerToken
        ZoomSDK.setCustomization(setUICustomization(idenfyLivenessUISettings, idenfyLivenessSettings, primaryColor, accentColor))

        initializeZoomSDK(activity)


    }


    private fun setUICustomization(idenfyLivenessUISettings: IdenfyLivenessUISettings?, idenfyLivenessSettings: IdenfyLivenessSettings?, primaryColor: Int, accentColor: Int): ZoomCustomization {
        val currentCustomization = ZoomCustomization()
        //core ui settings
        currentCustomization.showUserLockedScreen = idenfyLivenessSettings?.showUserLockedScreen
                ?: false
        currentCustomization.showPreEnrollmentScreen = idenfyLivenessSettings?.showPreEnrollmentScreen
                ?: false
        currentCustomization.showRetryScreen = idenfyLivenessSettings?.showUserLockedScreen ?: false

        val customUISettings = idenfyLivenessUISettings?.livenessCustomUISettings
        if (customUISettings != null) {
            customUISettings.showUserLockedScreen = idenfyLivenessSettings?.showUserLockedScreen
                    ?: false
            customUISettings.showPreEnrollmentScreen = idenfyLivenessSettings?.showPreEnrollmentScreen
                    ?: false
            customUISettings.showRetryScreen = idenfyLivenessSettings?.showUserLockedScreen ?: false
            return customUISettings
        }
        currentCustomization.mainForegroundColor = primaryColor
        val zoomFeedbackCustomization = ZoomFeedbackCustomization()
        val zoomFrameCustomization = ZoomFrameCustomization()
        zoomFrameCustomization.topMargin = 0
        zoomFrameCustomization.sizeRatio = 1f
        zoomFeedbackCustomization.backgroundColor = accentColor
        currentCustomization.zoomInterfaceEntryTransitionTime = 1.0
        currentCustomization.frameCustomization = zoomFrameCustomization
        currentCustomization.feedbackCustomization = zoomFeedbackCustomization
        currentCustomization.exitAnimationSuccessCustom = ZoomExitAnimationStyle.NONE
        currentCustomization.exitAnimationUnsuccessCustom = ZoomExitAnimationStyle.NONE

        if (idenfyLivenessUISettings != null) {
            val idenfyLivenessUIHelper = IdenfyLivenessUIHelper(idenfyLivenessUISettings)
            idenfyLivenessUIHelper.setupCustomLiveness(currentCustomization)
        } else {
            val defaultIdenfyLivenssUISettings = IdenfyLivenessUISettings()
            defaultIdenfyLivenssUISettings.livenessFeedbackBackgroundColor = accentColor
            defaultIdenfyLivenssUISettings.livenessFrameBackgroundColor = primaryColor
            defaultIdenfyLivenssUISettings.livenessIdentificationProgressStrokeWidth = 2
            defaultIdenfyLivenssUISettings.livenessIdentificationProgressRadialOffset = 6
            defaultIdenfyLivenssUISettings.livenessIdentificationProgressStrokeColor = accentColor
            defaultIdenfyLivenssUISettings.livenessMainBackgroundColor = primaryColor
            defaultIdenfyLivenssUISettings.livenessMainForegroundColor = primaryColor
            defaultIdenfyLivenssUISettings.livenessIdentificationOvalProgressColor1 = accentColor
            defaultIdenfyLivenssUISettings.livenessIdentificationOvalProgressColor2 = accentColor
            val idenfyLivenessUIHelper = IdenfyLivenessUIHelper(defaultIdenfyLivenssUISettings)
            idenfyLivenessUIHelper.setupCustomLiveness(currentCustomization)
        }
        return currentCustomization
    }

    private fun setCoreZoomSettings() {
        ZoomSDK.setAuditTrailType(ZoomAuditTrailType.HEIGHT_640)
        ZoomSDK.setFacemapEncryptionKey(sharedPrivateKey)
        ZoomSDK.setTimeBasedSessionImagesEnabled(true)
    }

    var isZoomInitialized = false
    fun initializeZoomSDK(activity: Activity?, shouldStartActivity: Boolean = false) {
        if (activity == null)
            return

        try {
            Class.forName(ConstsLiveliness.CONST_OPTIONAL_LIVELINESS_ENABLED)
        } catch (e: ClassNotFoundException) {
            return
        }


        ZoomSDK.initialize(activity, developerToken, object : ZoomSDK.InitializeCallback() {
            override fun onCompletion(successful: Boolean) {
                if (!successful) {
                    retryInit(activity, developerToken)
                } else {
                    isZoomInitialized = true
                    val zoomSDKStatus = ZoomSDK.getStatus(activity)
                    if (shouldStartActivity) {

                        if (zoomSDKStatus != ZoomSDKStatus.INITIALIZED) {
                            zoomSessionResponse.postValue(LivenessSessionNotSuccessfulWithoutRetries(true, null, false))
                            return
                        }

                        startActivityForResult(activity = activity)
                        return
                    }

                    if (zoomSDKStatus == ZoomSDKStatus.INITIALIZED) {
                        zoomSDKInicializationResponse.postValue(Result.Success(Pair(true, zoomSDKStatus.toString())))
                    } else {
                        if (zoomSDKStatus == ZoomSDKStatus.OFFLINE_SESSIONS_EXCEEDED)
                            zoomSDKInicializationResponse.postValue(Result.Success(Pair(false, zoomSDKStatus.toString())))
                        else {
                            retryInit(activity, developerToken)
                        }
                    }

                }
            }
        })
    }


    internal fun retryInit(activity: Activity?, developerToken: String) {
        if (inicializationCount < INICIALIZATION_COUNT_MAX) {
            initializeZoomSDK(activity)
        } else {
            inicializationCount = 0
            zoomSDKInicializationResponse.postValue(Result.Success(Pair(false, "")))
        }
        inicializationCount++
    }

    fun startActivityForResult(activity: FragmentActivity) {
        try {
            Class.forName(ConstsLiveliness.CONST_OPTIONAL_LIVELINESS_ENABLED)
        } catch (e: ClassNotFoundException) {
            return
        }
        if (ZoomSDK.getStatus(activity) == ZoomSDKStatus.DEVICE_LOCKED_OUT || ZoomSDK.getStatus(activity) == ZoomSDKStatus.OFFLINE_SESSIONS_EXCEEDED) {
            zoomErrorOccured.value = true
            return
        }
        val authenticationIntent = Intent(activity, ZoomVerificationActivity::class.java)
        activity.startActivityForResult(authenticationIntent, ZoomSDK.REQUEST_CODE_VERIFICATION)

    }

    fun startActivityForResult(activity: Activity) {
        try {
            Class.forName(ConstsLiveliness.CONST_OPTIONAL_LIVELINESS_ENABLED)
        } catch (e: ClassNotFoundException) {
            return
        }
        val authenticationIntent = Intent(activity, ZoomVerificationActivity::class.java)
        activity.startActivityForResult(authenticationIntent, ZoomSDK.REQUEST_CODE_VERIFICATION)

    }
}
