package com.idenfy.idenfySdk.backgrounduploading.data.repository

import android.content.Context
import android.util.Log
import com.idenfy.idenfySdk.Networking.APIService
import com.idenfy.idenfySdk.Networking.Models.PhotoPost
import com.idenfy.idenfySdk.Networking.Models.PresignedUrlModel
import com.idenfy.idenfySdk.backgrounduploading.data.datasource.BackgroundUploadingDataSource
import com.idenfy.idenfySdk.backgrounduploading.domain.repository.BackgroundUploadingRepository
import com.idenfy.idenfysdk.core.extensions.rxextensions.retryWhenError
import io.reactivex.Observable
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.disposables.CompositeDisposable
import io.reactivex.schedulers.Schedulers
import okhttp3.MediaType
import okhttp3.RequestBody
import retrofit2.HttpException
import java.io.File
import java.util.*

class BackgroundUploadingRepositoryImpl(val backgroundUploadingDataSourceImpl: BackgroundUploadingDataSource,
                                        val apiService: APIService)
    : BackgroundUploadingRepository {
    override fun uploadBackgroundPhotos(disposable: CompositeDisposable, token: String, key: String, context: Context) {
        val recordingStepsFile = backgroundUploadingDataSourceImpl.getUploadingFile(key)
        disposable.add(apiService
                .getBackgroundPhotoPostUrl(PhotoPost(token, key.toUpperCase()))
                .retryWhenError(5, 1, false)
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .flatMap { response -> Observable.just(response) }
                .subscribe({
                    val preSignedUrlModel = it
                    if (recordingStepsFile != null)
                        uploadBackgroundPhotosToServer(disposable, preSignedUrlModel, recordingStepsFile, key, context)
                }, {

                }, {

                }))

    }

    override fun getBackgroundPhotos(key: String): MutableList<ByteArray> {
        return backgroundUploadingDataSourceImpl.getBackgroundPhotos(key)
    }

    override fun setBackgroundPhotosFile(key: String, photos: File) {
        backgroundUploadingDataSourceImpl.setBackgroundPhotosFile(key, photos)
    }

    override fun addNewPhotoToBackgroundPhotos(photo: ByteArray, step: String) {
        backgroundUploadingDataSourceImpl.addNewPhotoToBackgroundPhotos(photo, step)

    }

    private fun uploadBackgroundPhotosToServer(disposable: CompositeDisposable, presignedUrlModel: PresignedUrlModel, backgroundPhotos: File, key: String, context: Context) {
        val hashMap = formUploadBackgroundPhotosData(presignedUrlModel)
        val fbody = RequestBody.create(MediaType.parse("text/plain"), backgroundPhotos)
        hashMap?.set("file", fbody)
        if (hashMap != null)
            uploadBackgroundPhotosToServer(disposable, presignedUrlModel, hashMap, key, context)

    }

    private fun uploadBackgroundPhotosToServer(disposable: CompositeDisposable, presignedUrlModel: PresignedUrlModel,
                                               hashMap: LinkedHashMap<String, RequestBody>, key: String, context: Context) {


        disposable.add(apiService.streamBackgroundPhotosObservable(hashMap, presignedUrlModel.url)
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .map {
                    if (it.raw() != null && !it.raw().isSuccessful) {
                        throw HttpException(it)
                    }
                    it
                }
                .retryWhenError(5, 1, true)
                .subscribe({ success ->
                    deleteUploadedZip(context, key)
                }, { error ->
                    deleteUploadedZip(context, key)
                }));


    }

    private fun deleteUploadedZip(context: Context, key: String) {
        val fileToDelete = File(context.cacheDir, "$key.zip")
        if(fileToDelete.exists()) {
            val deleted = fileToDelete.delete()
        }
    }


    private fun formUploadBackgroundPhotosData(responseData: PresignedUrlModel?): LinkedHashMap<String, RequestBody>? {
        if (responseData != null) {
            val requestBodyHashMap = LinkedHashMap<String, RequestBody>()

            val authorization = RequestBody.create(MediaType.parse("text/plain"), "application/zip")
            requestBodyHashMap["Content-Type"] = authorization

            val key = RequestBody.create(MediaType.parse("text/plain"), responseData.fields.key)
            requestBodyHashMap["key"] = key

            val awsAccessKeyId = RequestBody.create(MediaType.parse("text/plain"), responseData.fields.awsAccessKeyId)
            requestBodyHashMap["AWSAccessKeyId"] = awsAccessKeyId

            val signature = RequestBody.create(MediaType.parse("text/plain"), responseData.fields.signature)
            requestBodyHashMap["signature"] = signature

            val policy = RequestBody.create(MediaType.parse("text/plain"), responseData.fields.policy)
            requestBodyHashMap["policy"] = policy

            return requestBodyHashMap
        } else {
            return null
        }
    }

}