package com.idenfy.idenfySdk.core.datasource.repository

import android.app.Application
import android.arch.lifecycle.LiveData
import android.arch.lifecycle.MutableLiveData
import com.google.gson.GsonBuilder
import com.google.gson.JsonElement
import com.google.gson.JsonParser
import com.idenfy.docscanning.networking.models.DocDataKeyResponse
import com.idenfy.idenfySdk.CoreSdkInitialization.IdenfyController
import com.idenfy.idenfySdk.CoreSdkInitialization.IdenfyUserFlowController
import com.idenfy.idenfySdk.Networking.APIService
import com.idenfy.idenfySdk.Networking.Resource
import com.idenfy.idenfySdk.Networking.RetrofitFactoryKotlinOptional
import com.idenfy.idenfySdk.SdkResponseModels.AutenticationResult.AuthenticationResultResponseInternal
import com.idenfy.idenfySdk.SdkResponseModels.IdenfyErrorResponse
import com.idenfy.idenfySdk.countriesselection.ui.model.Country
import com.idenfy.idenfySdk.helpers.SingleLiveEvent
import com.idenfy.idenfySdk.identificationresults.presentation.state.IdentificationResultsUIViewModelState
import com.idenfy.idenfySdk.identificationresults.presentation.state.IdentificationResultsUploadedPhotosState
import com.idenfy.idenfySdk.identificationresults.presentation.uiviewmodel.IdentificationResultsUIViewModel
import com.idenfy.idenfySdk.core.presentation.uiviewmodel.InitialLoadingDataViewModel
import com.idenfy.idenfysdk.core.domain.networkrepo.ErrorsHelper
import com.idenfy.idenfysdk.core.extensions.livedataextensions.default
import com.idenfy.idenfysdk.core.extensions.rxextensions.retryWhenError
import com.idenfy.idenfysdk.core.internal.settings.IdenfyInternalSettings
import com.idenfy.idenfysdk.core.models.documentTypeData.DocumentType
import com.idenfy.idenfysdk.core.models.documentTypeData.Step
import com.idenfy.idenfysdk.core.networking.models.requestBodies.*
import com.idenfy.idenfysdk.core.networking.models.responseBodies.CountryCode
import com.idenfy.idenfysdk.core.networking.models.responseBodies.PartnerInfo
import com.idenfy.idenfysdk.core.security.VigenereCipher
import com.idenfySdk.R
import io.reactivex.Observable
import io.reactivex.Single
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.disposables.CompositeDisposable
import io.reactivex.functions.Function5
import io.reactivex.schedulers.Schedulers
import io.reactivex.subjects.ReplaySubject
import okhttp3.MediaType
import okhttp3.RequestBody
import okhttp3.ResponseBody
import retrofit2.HttpException
import java.io.File

class RepositoryKotlin(idenfyController: IdenfyController, application: Application, val apiService: APIService) {

    // identification results
    var identificationResultsUIViewModel = IdentificationResultsUIViewModel(LinkedHashMap())
    var identificationResultsUIViewModelLiveData = MutableLiveData<IdentificationResultsUIViewModelState<IdentificationResultsUIViewModel>>().default(IdentificationResultsUIViewModelState.Reset(IdentificationResultsUIViewModel(LinkedHashMap())))
    var uploadsStepsList: MutableList<String> = ArrayList<String>()

    //initialization
    var application: Application? = application

    //variables
    private var partnerInfo: PartnerInfo = PartnerInfo()
    private var authTokenRequest: AuthTokenRequest = AuthTokenRequest()

    fun getAuthTokenRequest(): AuthTokenRequest {
        return authTokenRequest
    }

    //Observables
    lateinit var setCountryObservable: Observable<ResponseBody>

    //triggers
    lateinit var startSessionSubjectTrigger: ReplaySubject<Boolean>
    lateinit var setDcumentTypeSubjectTrigger: ReplaySubject<Boolean>
    lateinit var zoomSDKSubjectTrigger: ReplaySubject<Boolean>
    lateinit var setDcumentIssuingCountryTrigger: ReplaySubject<Boolean>
    lateinit var facePhotoUploaedTrigger: ReplaySubject<Boolean>
    lateinit var docScanningUploadedTrigger: ReplaySubject<Boolean>
    lateinit var allPhotosUploadedTrigger: ReplaySubject<Boolean>
    lateinit var allPresignedModelTrigger: ReplaySubject<Boolean>
    var initialDataHasLoadedMutableLiveData = MutableLiveData<Boolean>()
    var countriesListMutableLiveData = MutableLiveData<Resource<List<Country>>>()

    val initialLoadingDataViewModel = MutableLiveData<com.idenfy.idenfysdk.core.networking.Resource<InitialLoadingDataViewModel>>()

    fun returnInitialLoadingDataViewModel(): MutableLiveData<com.idenfy.idenfysdk.core.networking.Resource<InitialLoadingDataViewModel>> {
        return initialLoadingDataViewModel
    }

    fun setupAuthTokenRequest(authToken: String) {
        this.authTokenRequest = AuthTokenRequest(authToken)
    }

    fun deInit() {
        initialDataHasLoadedMutableLiveData.postValue(false)
        countriesListMutableLiveData.postValue(com.idenfy.idenfySdk.Networking.Resource.loading(null))
        initialLoadingDataViewModel.value = com.idenfy.idenfysdk.core.networking.Resource.loading(null)
        idenfyInternalSettings = IdenfyInternalSettings()
    }

    fun fetchDocumentTypes(): Single<List<DocumentType>> {
        return apiService.getDocumentsTypesObservable(authTokenRequest.getAuthToken())
    }

    fun fetchGetCountryByIp(): Single<CountryCode> {
        return apiService.countryByIPAdObservable
    }

    fun getPartnerInfo(): PartnerInfo? {
        return partnerInfo
    }

    //TRIGGERS INIT
    fun setupDocumentTypeTrigger() {
        setDcumentTypeSubjectTrigger = ReplaySubject.create()
    }

    fun setupInitialTriggers() {
        setDcumentIssuingCountryTrigger = ReplaySubject.create()
        startSessionSubjectTrigger = ReplaySubject.create()
    }

    fun startSessionTriggerDone() {
        if (!startSessionSubjectTrigger.hasValue())
            startSessionSubjectTrigger.onNext(true)
    }

    fun setDocumentTypeTriggerDone() {

        setDcumentTypeSubjectTrigger.onNext(true)
    }

    fun setCountryTriggerDone() {
        setDcumentIssuingCountryTrigger.onNext(true)
    }

    fun setDocScanningTrigger(isActive:Boolean) {
        docScanningUploadedTrigger.onNext(isActive)
    }

    private fun setAllPhotosUploadedTriggerDone() {
        allPhotosUploadedTrigger.onNext(true)
    }

    private var idenfyErrorIdentifierLiveData = SingleLiveEvent<String>()

    fun returnIdenfyErrorIdentifierLiveData(): SingleLiveEvent<String> {
        return idenfyErrorIdentifierLiveData
    }

    fun getIdenfyErrorIdentifier(): LiveData<String> {
        return idenfyErrorIdentifierLiveData
    }

    var idenfyInternalSettings = IdenfyInternalSettings()

    fun setPartnerInfo(partnerInfo: PartnerInfo) {
        if (partnerInfo.getCountry() != null) {
            IdenfyUserFlowController.onIssuingCountrySelected(partnerInfo.country)
            setCountryTriggerDone()
        }

        this.partnerInfo = partnerInfo
    }

    fun fetchPartnerInfoSingle(): Single<PartnerInfo> {
        return apiService.getPartnerInfoSingle(authTokenRequest)
    }

    fun startSession(authToken: String, compositeDisposable: CompositeDisposable) {
        idenfyInternalSettings.isSessionsStarted = true
        val authTokenRequest = AuthTokenRequest(authToken)
        compositeDisposable.add(apiService.startSessionObservable(authTokenRequest)
                .retryWhenError(5, 1, false)
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .take(1)
                .subscribe({
                    startSessionTriggerDone()
                }, { throwable -> handleErrorInstance(throwable) }))
    }

    fun checkLiveness(livenessCheck: com.idenfy.idenfyliveness.LivenessCheck, compositeDisposable: CompositeDisposable) {
        updateCurrentStepsUploadingData(Step.LIVENESS.toString(), IdentificationResultsUploadedPhotosState.IdentificationResultsStatus.LOADING)
        compositeDisposable.add(facePhotoUploaedTrigger
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .concatMap { response ->
                    RetrofitFactoryKotlinOptional.makeRetrofitService()
                            .checkLivenessObservable(livenessCheck)
                            .retryWhenError(5, 1, false)
                            .subscribeOn(Schedulers.io())
                            .observeOn(AndroidSchedulers.mainThread())
                }.subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .take(1)
                .subscribe({ success ->
                    zoomSDKSubjectTrigger.onNext(true)
                    updateCurrentStepsUploadingData(Step.LIVENESS.toString(), IdentificationResultsUploadedPhotosState.IdentificationResultsStatus.SUCCESS)
                }, { throwable -> handleErrorInstance(throwable) }))
    }

    fun setIssuingCountry(countrySetRequest: CountrySetRequest, compositeDisposable: CompositeDisposable) {
        setCountryObservable = apiService
                .setIssuingCountryObservable(countrySetRequest)
        compositeDisposable.add(setCountryObservable
                .retryWhenError(5, 1, false)
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe({ success ->
                    IdenfyUserFlowController.onIssuingCountrySelected(countrySetRequest.country)
                    setCountryTriggerDone()
                }, { throwable -> handleErrorInstance(throwable) }))
    }

    fun startObservingUploads(stepList: List<String>) {
        val copiedList = stepList.toMutableList()
        uploadsStepsList = copiedList
        zoomSDKSubjectTrigger = ReplaySubject.create()
        docScanningUploadedTrigger = ReplaySubject.create()
        facePhotoUploaedTrigger = ReplaySubject.create()
        allPhotosUploadedTrigger = ReplaySubject.create()
        allPresignedModelTrigger = ReplaySubject.create()
        responseList = arrayListOf()
        hashCodesList = arrayListOf()


        if (!shouldSetupLiveness(stepList)) {
            zoomSDKSubjectTrigger.onNext(true)
        }
        else{
            uploadsStepsList.add(Step.LIVENESS.toString())
        }
        if (!shouldSetupDocScanning(stepList)) {
            setDocScanningTrigger(true)
        }


    }

    fun observeThenToStartProcessing(): Observable<Boolean> {
        return Observable.zip(setDcumentTypeSubjectTrigger, setDcumentIssuingCountryTrigger, zoomSDKSubjectTrigger,
                allPhotosUploadedTrigger, docScanningUploadedTrigger,
                Function5<Boolean, Boolean, Boolean, Boolean, Boolean, Boolean> { setDcumentTypeSubjectTrigger, setDcumentIssuingCountryTrigger,
                                                                                  zoomSDKSubjectTrigger, allPhotosUploadedTrigger, docScanningTrigger ->
                    indicate(setDcumentTypeSubjectTrigger, setDcumentIssuingCountryTrigger,
                            zoomSDKSubjectTrigger, allPhotosUploadedTrigger, docScanningTrigger)
                })
    }

    private fun indicate(setDcumentTypeSubjectTrigger: Boolean, setDcumentIssuingCountryTrigger: Boolean,
                         zoomSDKSubjectTrigger: Boolean, allPhotosUploadedTrigger: Boolean, docScanningTrigger: Boolean): Boolean {
        if (setDcumentTypeSubjectTrigger && setDcumentIssuingCountryTrigger && zoomSDKSubjectTrigger &&
                allPhotosUploadedTrigger && docScanningTrigger) {
            return true
        }
        return false
    }

    fun checkIdentificationResults(): Observable<AuthenticationResultResponseInternal> {
        return apiService.checkAuthenticationStatusObservable(authTokenRequest)
    }


    //ERROR HANDLING
    fun handleErrorInstance(error: Throwable) {
        if (error is HttpException) {
            try {
                val errorCode = error.code()
                val errorBody = error.response().errorBody()
                handleError(errorCode, errorBody)
            } catch (e: Exception) {
                idenfyErrorIdentifierLiveData.postValue(ErrorsHelper.MALFORMED_JSON_IDENTIFIER)
            }

        } else {
            idenfyErrorIdentifierLiveData.postValue(ErrorsHelper.MALFORMED_JSON_IDENTIFIER)
        }
    }

    private fun handleError(code: Int, errorBody: ResponseBody?) {
        if (code in 300..499) {
            val parser = JsonParser()
            var mJson: JsonElement? = null
            try {
                mJson = parser.parse(errorBody?.string())
                val gson = GsonBuilder().setLenient().create()
                val errorResponse = gson.fromJson(mJson, IdenfyErrorResponse::class.java)
                idenfyErrorIdentifierLiveData.postValue(errorResponse.identifier)
            } catch (ex: Exception) {
                idenfyErrorIdentifierLiveData.postValue(ErrorsHelper.MALFORMED_JSON_IDENTIFIER)
            }

        } else if (code >= 500) {
            idenfyErrorIdentifierLiveData.postValue(ErrorsHelper.IDENFY_SERVER_ERROR_IDENTIFIER)
        }
    }

    private fun shouldSetupLiveness(stepList: List<String>): Boolean {
        if (!partnerInfo.zoomLiveliness) {
            return false
        }
        return stepList.contains(Step.FACE.name)
    }

    private fun shouldSetupDocScanning(stepList: List<String>): Boolean {
        if (!partnerInfo.smartEnginesMobile) {
            return false
        }
        return stepList.contains(Step.FRONT.name)

    }


    //Sets document type
    fun setupDocument(documentTypeRequest: DocumentTypeRequest, compositeDisposable: CompositeDisposable, hasPartnerEnabledZoom: Boolean) {
        compositeDisposable.add(startSessionSubjectTrigger
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .concatMap { response ->
                    apiService
                            .setDocumentTypeObservable(documentTypeRequest)
                            .retryWhenError(5, 1, false)
                            .subscribeOn(Schedulers.io())
                            .observeOn(AndroidSchedulers.mainThread())
                }.subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe({ success ->
                    if (documentTypeRequest.documentType != null)
                        IdenfyUserFlowController.onDocumentSelected(documentTypeRequest.documentType!!)
                    setDocumentTypeTriggerDone()
                }, { throwable ->
                    handleErrorInstance(throwable)
                })
        )
    }

    fun getS3UploadData(compositeDisposable: CompositeDisposable, s3RequestData: S3RequestData, image: File, base64: String, stepList: MutableList<String>, isImageUploaded: Boolean) {
        compositeDisposable.add(startSessionSubjectTrigger
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe {
                    if (it) {
                        if (isImageUploaded) {
                            apiService.getS3UploadDataForUpload(s3RequestData)
                                    .retryWhenError(5, 1, false)
                                    .subscribeOn(Schedulers.io())
                                    .observeOn(AndroidSchedulers.mainThread())
                                    .subscribe({ presignedUrlModel ->
                                        formData(compositeDisposable, presignedUrlModel, stepList, image, base64, s3RequestData.step, isImageUploaded)
                                    }, { error -> handleErrorInstance(error) }
                                    )
                        } else {
                            apiService.getS3UploadDataForStream(s3RequestData)
                                    .retryWhenError(5, 1, true)
                                    .subscribeOn(Schedulers.io())
                                    .observeOn(AndroidSchedulers.mainThread())
                                    .subscribe({ presignedUrlModel ->
                                        formData(compositeDisposable, presignedUrlModel, stepList, image, base64, s3RequestData.step, isImageUploaded)
                                    }, { error -> handleErrorInstance(error) }
                                    )
                        }
                    }
                })
    }


    private fun uploadDocumentPhoto(compositeDisposable: CompositeDisposable,
                                    url: String,
                                    linkedHashMap: LinkedHashMap<String, RequestBody>,
                                    checkList: List<String>,
                                    image: File,
                                    base64: String,
                                    step: String, isImageUploaded: Boolean) {

        val hashcode = linkedHashMap.hashCode()
        hashCodesList.add(hashcode)
        linkedHashMap.put("file", RequestBody.create(MediaType.parse("image/*"), image))
        val observable = apiService.uploadPhotoToS3(linkedHashMap, url)
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .map {
                    if (it.raw() != null && !it.raw().isSuccessful) {
                        throw HttpException(it)
                    }
                    it
                }
                .retryWhenError(5, 1, true)
                .subscribe({

                    response ->
                    if (response.raw().code() in 200..300)
                        getUploadedPhotoResponse(response, checkList, hashcode, step)
                    else uploadDocumentPhotoLegacy(compositeDisposable, checkList, base64, step, isImageUploaded)


                }, { error ->
                    uploadDocumentPhotoLegacy(compositeDisposable, checkList, base64, step, isImageUploaded)
                })


        compositeDisposable.add(observable)
    }

    fun uploadDocumentPhotoLegacy(compositeDisposable: CompositeDisposable,
                                  checkList: List<String>,
                                  base64: String,
                                  step: String, isImageUploaded: Boolean) {

        val hashcode = base64.hashCode()
        hashCodesList.add(hashcode)

        val streamPost = StreamPost(authTokenRequest.authToken, base64, step)
        if (isImageUploaded) {
            val observable = apiService.streamUploadPhotoNewObservable(streamPost)
                    .retryWhenError(5, 1, false)
                    .subscribeOn(Schedulers.io())
                    .observeOn(AndroidSchedulers.mainThread())
                    .subscribe({ success ->
                        getUploadedPhotoResponse(success, checkList, hashcode, step)
                    }, { throwable ->
                        handleErrorInstance(throwable)
                    })
            compositeDisposable.add(observable)
        } else {
            val observable = apiService.streamPhotoNewObservable(streamPost)
                    .retryWhenError(5, 1, false)
                    .subscribeOn(Schedulers.io())
                    .observeOn(AndroidSchedulers.mainThread())
                    .subscribe({ success ->
                        getUploadedPhotoResponse(success, checkList, hashcode, step)
                    }, { throwable ->
                        handleErrorInstance(throwable)
                    })
            compositeDisposable.add(observable)
        }
    }


    private var responseList = ArrayList<Any>()
    var hashCodesList = ArrayList<Int>()
    private fun getUploadedPhotoResponse(response: Any, checkList: List<String>, hashcode: Int, step: String) {

        if (!hashCodesList.contains(hashcode)) {
            return
        }
        responseList.add(response)

        if (step.equals(Step.FACE.name, ignoreCase = true)) {
            facePhotoUploaedTrigger.onNext(true)
        }
        if (responseList.size == checkList.size) {
            setAllPhotosUploadedTriggerDone()
        }

        updateCurrentStepsUploadingData(step, IdentificationResultsUploadedPhotosState.IdentificationResultsStatus.SUCCESS)
    }

    fun updateCurrentStepsUploadingData(step:String, status: IdentificationResultsUploadedPhotosState.IdentificationResultsStatus)
    {
        var stateData = R.string.idenfy_empty_string
        when(step){
            Step.PASSPORT_COVER.toString()->{
                stateData = R.string.idenfy_identification_results_title_additional_info_uploading_passport_cover_photo
            }
            Step.UTILITY_BILL.toString()->{
                stateData = R.string.idenfy_identification_results_title_additional_info_uploading_utility_bill_photo
            }
            Step.FRONT.toString()->{
                stateData = R.string.idenfy_identification_results_title_additional_info_uploading_front_document_photo
            }
            Step.BACK.toString()->{
                stateData = R.string.idenfy_identification_results_title_additional_info_uploading_back_document_photo
            }
            Step.FACE.toString()->{
                stateData = R.string.idenfy_identification_results_title_additional_info_uploading_face_document_photo
            }
            Step.LIVENESS.toString()->{
                stateData = R.string.idenfy_identification_results_title_additional_info_uploading_face_document_photo
            }

        }

        when(status)
        {
            IdentificationResultsUploadedPhotosState.IdentificationResultsStatus.LOADING -> {
                identificationResultsUIViewModel.uploadedPhotosStatesDictionary[step] = IdentificationResultsUploadedPhotosState.Loading(stateData)
            }
            IdentificationResultsUploadedPhotosState.IdentificationResultsStatus.SUCCESS -> {
                stateData = R.string.idenfy_empty_string
                identificationResultsUIViewModel.uploadedPhotosStatesDictionary[step] = IdentificationResultsUploadedPhotosState.Success(stateData)

            }
            IdentificationResultsUploadedPhotosState.IdentificationResultsStatus.ERROR ->{

            }
        }
        val stepCount = uploadsStepsList.size
        val percentageIncrease = 100 / stepCount
        var uploadedSuccessfully = 0
        var progressOfSuccess = 0
        for (key in identificationResultsUIViewModel.uploadedPhotosStatesDictionary.keys) {
            val dictionaryData = identificationResultsUIViewModel.uploadedPhotosStatesDictionary[key]
            when(key){
                Step.PASSPORT_COVER.toString(), Step.FRONT.toString(), Step.BACK.toString(), Step.FACE.toString(),
                Step.LIVENESS.toString(), Step.UTILITY_BILL.toString() -> {
                    when(dictionaryData){
                        is IdentificationResultsUploadedPhotosState.Success -> {
                            if(stepCount==identificationResultsUIViewModel.uploadedPhotosStatesDictionary.keys.size) {
                                uploadedSuccessfully+=1
                                progressOfSuccess += percentageIncrease
                            }
                        }
                    }
                }
            }
        }
        if(uploadedSuccessfully==stepCount)
        {
            identificationResultsUIViewModelLiveData.postValue(IdentificationResultsUIViewModelState.UploadsFinished(identificationResultsUIViewModel.copy(progressOfSuccess=progressOfSuccess)))
        }
        else{
            identificationResultsUIViewModelLiveData.postValue(IdentificationResultsUIViewModelState.InProgress(identificationResultsUIViewModel.copy(progressOfSuccess=progressOfSuccess)))
        }
    }


    private fun formData(compositeDisposable: CompositeDisposable, presignedUrlModel: PresignedUrlModel, stepList: List<String>, image: File, base64: String, step: String, isImageUploaded: Boolean) {
        uploadDocumentPhoto(compositeDisposable,
                presignedUrlModel.url,
                getPresignedUrlModelHashMap(presignedUrlModel)!!,
                stepList, image, base64, step, isImageUploaded)
    }


    private fun getPresignedUrlModelHashMap(presignedUrlModel: PresignedUrlModel?): LinkedHashMap<String, RequestBody>? {
        if (presignedUrlModel != null) {
            val requestBodyHashMap = LinkedHashMap<String, RequestBody>()

            requestBodyHashMap.put("Content-Type", RequestBody.create(MediaType.parse("text/plain"), "image/png"))
            requestBodyHashMap.put("key", RequestBody.create(MediaType.parse("text/plain"), presignedUrlModel.fields.key))
            requestBodyHashMap.put("AWSAccessKeyId", RequestBody.create(MediaType.parse("text/plain"),
                    presignedUrlModel.fields.AWSAccessKeyId))
            requestBodyHashMap.put("signature", RequestBody.create(MediaType.parse("text/plain"), presignedUrlModel.fields.signature))
            requestBodyHashMap.put("policy", RequestBody.create(MediaType.parse("text/plain"), presignedUrlModel.fields.policy))

            return requestBodyHashMap;

        } else {
            return null
        }
    }

    fun uploadDocReadingData(SDKLifecycleDisposable: CompositeDisposable, data: String, salt: String):Observable<ResponseBody> {
       return getDocDataKey()
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .flatMap { docDataKeyResponse ->
                    com.idenfy.docscanning.networking.RetrofitFactory.makeRetrofitService().uploadDocReadingInfo(docDataKeyResponse.est,
                            authTokenRequest.authToken,
                            RequestBody.create(MediaType.parse("text/plain"),
                                    createData(data, docDataKeyResponse.est,
                                            authTokenRequest.authToken, salt)))
                            .retryWhenError(5, 1, false)
                            .subscribeOn(Schedulers.io())
                            .observeOn(AndroidSchedulers.mainThread())
                }
    }

    private fun createData(docData: String, est: String, authToken: String, salt: String): String {
        var salt = salt
        salt = est + salt + authToken

        return VigenereCipher.encrypt(docData, salt)
    }

    private fun getDocDataKey(): Observable<DocDataKeyResponse> {
        return com.idenfy.docscanning.networking.RetrofitFactory.makeRetrofitService().getUploadingDocReadingKeys()
    }
}



