package huytranq.resource;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.drawable.Drawable;
import android.support.v4.content.ContextCompat;
import android.util.Log;

/**
 * Created by huytr on 28-01-2016.
 */
public final class Resource {

    //  Enums for resources' types
    public enum Type {
        BOOLEAN("bool") ,
        COLOR("color") ,
        DIMEN("dimen") ,
        ID("id") ,
        INTEGER("integer") ,
        ARRAY("array") ,
        STRING("string") ,
        DRAWABLE("drawable")
        ;

        private final String name;

        Type(String name) {
            this.name = name;
        }

        public boolean equals(Type type) {
            return name.equals(type.name);
        }

        public boolean equals(String resourceType) {
            return name.equals(resourceType);
        }

        @Override
        public String toString() {
            return name;
        }
    }

    //  Fields
    private static Resource resource;
    private Context context;

    //  Methods

    private Resource(Context context) {
        this.context = context.getApplicationContext();
    }

    public synchronized static Resource from(Context context) {
        if (resource == null) {
            resource = new Resource(context);
        }
        return resource;
    }

    public int getResourceId(String name ,
                             Type type) {
        String packageName = context.getPackageName();
        return context.getResources().getIdentifier(name , type.toString() , packageName);
    }

    public int getColorId(String name) {
        int result = -1;
        String except = null;
        try {
            result = R.color.class.getField(name).getInt(null);
        }
        catch (Exception exception) {
            Log.d(exception.getClass().getName() , "\"" + exception.getMessage() + "\" caused by \"" + exception.getCause() + "\"");
        }
        return result;
    }

    public String getString(String name) {
        return context.getResources().getString(getResourceId(name , Type.STRING));
    }

    public String getString(int id) {
        return context.getResources().getString(id);
    }

    public int getInteger(String name) {
        return context.getResources().getInteger(getResourceId(name, Type.INTEGER));
    }

    public int getInteger(int id) {
        return context.getResources().getInteger(id);
    }

    public String[] getStringArray(String name) {
        return context.getResources().getStringArray(getResourceId(name, Type.ARRAY));
    }

    public String[] getStringArray(int id) {
        return context.getResources().getStringArray(id);
    }

    public int[] getIntegerArray(String name) {
        return context.getResources().getIntArray(getResourceId(name, Type.ARRAY));
    }

    public int[] getIntegerArray(int id) {
        return context.getResources().getIntArray(id);
    }

    public boolean getBoolean(String name) {
        return context.getResources().getBoolean(getResourceId(name, Type.BOOLEAN));
    }

    public boolean getBoolean(int id) {
        return context.getResources().getBoolean(id);
    }

    public TypedArray getTypedArray(String name) {
        return context.getResources().obtainTypedArray(getResourceId(name, Type.ARRAY));
    }

    public TypedArray getTypedArray(int id) {
        return context.getResources().obtainTypedArray(id);
    }

    public float getDimension(String name) {
        return context.getResources().getDimension(getResourceId(name, Type.DIMEN));
    }

    public float getDimension(int id) {
        return context.getResources().getDimension(id);
    }

    public Drawable getDrawable(String name) {
        return ContextCompat.getDrawable(context, getResourceId(name, Type.DRAWABLE));
    }

    public Drawable getDrawable(int id) {
        return ContextCompat.getDrawable(context , id);
    }
}
