(ns daifu.diagnosis.summary.stat
  (:require [daifu.diagnosis.result :as result])
  (:import daifu.diagnosis.result.Result
           java.util.List
           clojure.lang.APersistentMap))

(declare sum)

(defmulti ->stat
  "creates a stat (of type Number) from various data types
   (->stat nil) => 0
   (->stat false) => 0
   (->stat true) => 1
   (->stat 10) => 10
   (->stat [1 2 3]) => 6
   (->stat (result/result [1 2 3])) => 6
   (->stat {:suggested \"(+ 1 1)\" :actual \"(inc 1)\"}) => 1"
  {:added "0.2"}
  type)

(defmethod ->stat nil
  [obj]
  0)

(defmethod ->stat Number
  [obj]
  obj)

(defmethod ->stat Boolean
  [obj]
  (if (true? obj) 1 0))

(defmethod ->stat Result
  [{:keys [value raw] :as result}]
  (cond value
        (->stat value)

        raw (->stat raw)

        :else 1))

(defmethod ->stat List
  [obj]
  (sum obj))

(defmethod ->stat APersistentMap
  [obj]
  (->stat (result/result obj)))

(defn sum
  "calculates the sum of all data points in a list
   (sum [true false true]) => 2
   (sum [1 2 3 4]) => 10"
  {:added "0.2"}
  [xs]
  (apply + (map ->stat xs)))

(defn average
  "calculates the average of data points in a list
   (average [true false true false]) => 1/2
   (average [1 2 3 4 5]) => 3"
  {:added "0.2"}
  [xs]
  (if (empty? xs)
    0
    (/ (sum xs) (count xs))))
