/*
 * File:                EvaluatorBasedCognitiveModule.java
 * Authors:             Justin Basilico, Kevin R. Dixon, and Zachary Benz
 * Company:             Sandia National Laboratories
 * Project:             Cognitive Framework Lite
 *
 * Copyright June 21, 2007, Sandia Corporation.  Under the terms of Contract
 * DE-AC04-94AL85000, there is a non-exclusive license for use of this work by
 * or on behalf of the U.S. Government. Export of this program may require a
 * license from the United States Government. See CopyrightHistory.txt for
 * complete details.
 *
 *
 */

package gov.sandia.cognition.framework.learning;

import gov.sandia.cognition.framework.learning.converter.CogxelConverter;
import gov.sandia.cognition.framework.CognitiveModel;
import gov.sandia.cognition.framework.CognitiveModelState;
import gov.sandia.cognition.framework.CognitiveModuleState;
import gov.sandia.cognition.framework.CogxelState;
import gov.sandia.cognition.evaluator.Evaluator;
import gov.sandia.cognition.framework.concurrent.AbstractConcurrentCognitiveModule;

/**
 * The EvaluatorBasedCognitiveModule implements a CognitiveModule that wraps an
 * Evaluator object.
 *
 * @param <InputType> Input type of the embedded Evaluator
 * @param <OutputType> Output type of the embedded Evaluator
 * @author Justin Basilico
 * @author Kevin R. Dixon
 * @author Zachary Benz
 * @since  2.0
 */
public class EvaluatorBasedCognitiveModule<InputType, OutputType>    
    extends AbstractConcurrentCognitiveModule
{
    
    /**
     * Name to assign to the module
     */
    private String name;    
    
    /** The module settings. */
    private EvaluatorBasedCognitiveModuleSettings<InputType, OutputType> 
            settings;
    
    /** 
     * A place to temporarily store the input read in by a call to readState;
     * this temporary store is blown away as soon as it used by evaluate,
     * because we NEVER retain state interally across module update cycles
     */
    protected InputType input;
    
    /** 
     * A place to temporarily store the output generated by a call to evaluate;
     * this temporary store is blown away as soon as it used by evaluate,
     * because we NEVER retain state interally across module update cycles
     */
    protected OutputType output;

    /**
     * Default name given to modules of this type
     */
    public static final String DEFAULT_NAME = "Evaluator-based Cognitive Module";
    
    /**
     * Creates a new instance of EvaluatorBasedCognitiveModule.
     *
     * @param  model The model to create the module for.
     * @param  settings The settings of the module.
     * @param name 
     * High-level name of this module
     */
    public EvaluatorBasedCognitiveModule(
        CognitiveModel model,
        EvaluatorBasedCognitiveModuleSettings<InputType, OutputType> settings,
        String name )
    {
        super();
        
        this.setSettings(settings.clone());
        this.getInputConverter().setSemanticIdentifierMap(
            model.getSemanticIdentifierMap());
        this.getOutputConverter().setSemanticIdentifierMap(
            model.getSemanticIdentifierMap());
        this.setName( name );
    }
    
    /**
     * {@inheritDoc}
     *
     * @param  modelState {@inheritDoc}
     * @return {@inheritDoc}
     */
    public CognitiveModuleState initializeState(
        CognitiveModelState modelState)
    {
        // This module has no state.
        return null;
    }
        
    /**
     * {@inheritDoc}
     * 
     * @param modelState {@inheritDoc}
     * @param previousModuleState {@inheritDoc}
     * @since 2.0
     */
    public void readState(CognitiveModelState modelState,
        CognitiveModuleState previousModuleState)
    {
        // Get the cogxels from the model state.
        CogxelState cogxels = modelState.getCogxels();
        
        // Convert the Cogxels to the input type.
        this.input = this.getInputConverter().fromCogxels(cogxels);
    }

    /**
     * {@inheritDoc}
     * @since 2.0
     */
    public void evaluate()
    {
        // Evaluate the input type.
        this.output = this.getEvaluator().evaluate(this.input);
        
        // Done with temporarily held input, so blow it away (we NEVER retain 
        // state locally across module update cycles)
        this.input = null;
    }

    /**
     * {@inheritDoc}
     * 
     * @param modelState {@inheritDoc}
     * @return {@inheritDoc}
     * @since 2.0
     */
    public CognitiveModuleState writeState(CognitiveModelState modelState)
    {
        // Get the cogxels from the model state.
        CogxelState cogxels = modelState.getCogxels();
        
        // Convert the output type back to Cogxels.
        this.getOutputConverter().toCogxels(this.output, cogxels);
        
        // Done with temporarily held output, so blow it away (we NEVER retain 
        // state locally across module update cycles)
        this.output = null;
        
        // This module has no state.
        return null;
    }           
    
    /**
     * {@inheritDoc}
     *
     * @return {@inheritDoc}
     */
    public String getName()
    {
        return this.name;
    }
    
    
    /**
     * Sets the name of the module
     * @param name 
     * Name to assign to the module
     */
    public void setName(
        final String name )
    {
        this.name = name;
    }
    
    
    /**
     * {@inheritDoc}
     *
     * @return {@inheritDoc}
     */
    public EvaluatorBasedCognitiveModuleSettings<InputType, OutputType> 
        getSettings()
    {
        return this.settings;
    }
    
    /**
     * Sets the settings of the module.
     *
     * @param  settings The module's settings.
     */
    protected void setSettings(
        EvaluatorBasedCognitiveModuleSettings<InputType, OutputType> settings)
    {
        this.settings = settings;
    }
    
    /**
     * Gets the underlying evaluator that is being wrapped in this module.
     * 
     * @return The underlying evaluator that is being wrapped in this module.
     */
    public Evaluator<? super InputType, ? extends OutputType> getEvaluator()
    {
        return this.getSettings().getEvaluator();
    }
    
    /**
     * Gets the input converter that is used to map from Cogxels to the 
     * InputType.
     *
     * @return The input converter.
     */
    public CogxelConverter<InputType> getInputConverter()
    {
        return this.getSettings().getInputConverter();
    }
    
    /**
     * Gets the output converter that is used to map from the OutputType to
     * Cogxels.
     *
     * @return The output converter.
     */
    public CogxelConverter<OutputType> getOutputConverter()
    {
        return this.getSettings().getOutputConverter();
    }
}
