package gov.raptor.gradle.plugins.jbossmodules

import groovy.transform.ToString
import org.gradle.api.internal.project.ProjectInternal
import org.gradle.api.tasks.Input
import org.gradle.api.tasks.Internal
import org.gradle.api.tasks.OutputDirectory
import org.gradle.util.ConfigureUtil

/**
 * The extension to hold our configuration values.
 */
@ToString(includeNames = true, includePackage = false)
class JbossModulesExtension {

    private final ProjectInternal project

    /**
     * The default list of locations to search for modules descriptors.
     */
    @Internal
    private static final List<String> defaultModulesDirs = [
            "src/etc/jboss-modules",
            "src/main/config/jboss-modules",
            "src/main/resources/META-INF/jboss-modules"
    ]

    @Internal
    boolean verbose = false

    /**
     * The name of the default slot to write to. See <code>slotStrategy</code>.
     *
     * @since 1.0
     * @see #slotStrategy
     */
    @Input
    String defaultSlot = 'main'

    /**
     * The name of the slot strategy to us. If not specified, the major version of
     * the dependency will be used as slot value.
     * <p>
     * Possible values are:
     * </p>
     * <table>
     * <tr>
     * <th>value</th>
     * <th>description</th>
     * </tr>
     * <tr>
     * <td>version-major</td>
     * <td>The slot has the major number of the version. The
     * <code>defaultSlot</code> is prepended, if not set to <code>main</code>
     * (e.g. <code>defaultSlot</code>=prodx and version 1.2.3 then the slot will
     * be named prodx1.</td>
     * </tr>
     * <tr>
     * <td>main</td>
     * <td>The slot has the name as given with <code>defaultSlot</code>.</td>
     * </tr>
     * </table>
     *
     * @since 1.0
     */
    @Input
    String slotStrategy = 'main'

    /**
     * The root directories to search for modules XML files that contain module
     * descriptors.
     * <p>
     * If not specified, the default locations
     * <code>src/main/config/jboss-modules</code>,
     * <code>src/main/resources/META-INF/jboss-modules</code>, and
     * <code>src/etc/jboss-modules</code> is probed and - if exists - are
     * appended.
     * </p>
     * <p>
     * You may want to use only one of the locations given above. Use
     * <code>config</code> if you do not want to have the configuration files
     * included. Use <code>resources/META-INF</code> if they should and use
     * <code>etc</code> if they should not, but be stored outside the
     * <code>main</code> folder.
     * </p>
     *
     * <pre>
     * &lt;modules&gt;
     *   &lt;dir&gt;src/etc/jboss-modules&lt;/dir&gt;
     * &lt;/modules&gt;
     * </pre>
     *
     * @since 1.0
     */
    @Input
    List<String> modules = defaultModulesDirs

    /**
     * Specifies the name of a module to add as a dependency on every module generated.
     */
    @Input
    String globalModuleDependency

    private List<File> compileDependencies

    /**
     * The folder to write the module structure to.
     */
    @OutputDirectory
    File targetFolder

    @Internal
    DependencyExcludes dependencyExcludes = new DependencyExcludes()

    JbossModulesExtension(ProjectInternal project) {
        this.project = project
        targetFolder = new File(project.buildDir, "/jboss-modules")
    }

    /**
     * The 'compile' dependencies are the primary input to this process, so if they change, we need to rebuild.
     *
     * @return The list of 'compile' dependencies
     */
    @SuppressWarnings("GroovyUnusedDeclaration")
    @Input
    List<File> getCompileDependencies() {
        // Calculate on first request
        if (!compileDependencies) {
            compileDependencies = project.configurations.compile.asList()
        }
        return compileDependencies
    }

    /**
     * Apply the given closure to configure our dependency excludes
     */
    @SuppressWarnings("GroovyUnusedDeclaration")
    void dependencyExcludes(Closure configureClosure) {
        ConfigureUtil.configure(configureClosure, dependencyExcludes)
    }

    /**
     * @return {@code true} if the configuration didn't specify any modules directories
     */
    boolean usingDefaultModules() {
        return modules == defaultModulesDirs
    }
}
