package gov.raptor.gradle.plugins.jbossmodules

import de.smartics.maven.plugin.jboss.modules.descriptor.ModuleDescriptor
import de.smartics.maven.plugin.jboss.modules.descriptor.ModulesDescriptor
import de.smartics.maven.plugin.jboss.modules.domain.ExecutionContext
import de.smartics.maven.plugin.jboss.modules.domain.ModuleBuilder
import de.smartics.maven.plugin.jboss.modules.domain.ModuleMap
import de.smartics.maven.plugin.jboss.modules.domain.SlotStrategy
import de.smartics.maven.plugin.jboss.modules.parser.ModulesXmlLocator
import org.gradle.api.DefaultTask
import org.gradle.api.artifacts.Configuration
import org.gradle.api.artifacts.ModuleVersionIdentifier
import org.gradle.api.artifacts.ResolvedArtifact
import org.gradle.api.artifacts.ResolvedDependency
import org.gradle.api.artifacts.ResolvedModuleVersion
import org.gradle.api.internal.artifacts.ResolvedConfigurationIdentifier
import org.gradle.api.tasks.Nested
import org.gradle.api.tasks.TaskAction
import org.gradle.tooling.BuildException

/**
 * Task to generate the Jboss Modules definitions to the target directory.
 */
class GenerateJbossModulesTask extends DefaultTask {

    @Nested
    JbossModulesExtension extension = project.extensions.getByType(JbossModulesExtension)

    /**
     * The linear (flattened) list of all modules descriptors.
     */
    private List<ModuleDescriptor> allModules

    /**
     * The configuration we'll be processing for dependencies.
     */
    private Configuration compileConfiguration

    GenerateJbossModulesTask() {
        extension = project.extensions.getByType(JbossModulesExtension)
    }

    @SuppressWarnings("GroovyUnusedDeclaration")
    @TaskAction
    protected void generate() {

        extension = project.extensions.getByType(JbossModulesExtension)
        compileConfiguration = project.configurations.getByName('compile')

        if (extension.verbose) project.logger.quiet("Configuration: " + extension)

        allModules = findModuleDescriptors()

        Set<ResolvedDependency> prunedDependencies =
                new DependencyPruner(project, extension.dependencyExcludes.excludes, allModules).getPrunedDependencies()

        runModuleCreation(prunedDependencies)
    }

    /**
     * Using the resolved and pruned dependencies, run the module creation.
     *
     * @param prunedDependencies Set of pruned dependencies to process into modules
     */
    private void runModuleCreation(final Set<ResolvedDependency> prunedDependencies) {

        // Delete the target directory so we start clean
        final targetFolder = extension.getTargetFolder()
        if (targetFolder.exists()) {
            assert targetFolder.deleteDir() // Nuke the whole thing
        }

        assert !targetFolder.exists()

        final ExecutionContext context = createContext(prunedDependencies)

        context.getModuleMap().toMap().each { module, dependencies ->
            try {
                new ModuleBuilder(context, module, dependencies).create()
            }
            catch (IOException e) {
                throw new BuildException("Cannot write module '${module.getName()}'.", e)
            }
        }
    }

    private ExecutionContext createContext(final Set<ResolvedDependency> prunedDependencies) {
        // Convert the gradle dependencies into mapped dependencies to integrate with the old code
        List<MappedDependency> dependencies = prunedDependencies.collect { new MappedDependency(it) }

        final ExecutionContext.Builder builder = new ExecutionContext.Builder()
        builder.with(logger)
        builder.withTargetFolder(extension.targetFolder)
        builder.with(new DirectDependencyResolver(extension.dependencyExcludes.excludes))
        builder.with(SlotStrategy.fromString(extension.slotStrategy))
        builder.withDefaultSlot(extension.defaultSlot)

        final ModuleMap moduleMap = new ModuleMap(allModules, expandClassifiers(dependencies))
        builder.with(moduleMap)

        if (extension.globalModuleDependency) {
            builder.withGlobalModuleDependency(extension.globalModuleDependency)
        }

        if (extension.verbose) {
            logger.quiet("Modules:\n" + moduleMap.toString())
        }

        return builder.build()
    }

    /**
     * Gradle and maven handle classifiers differently.  In maven, since you can only have 1 artifact per dependency,
     * each classifier shows up as a unique dependency.  In gradle, which allows multiple artifacts for a dependency,
     * you get a single dependency and multiple resolved artifacts.  So, to interface with the old maven-based code,
     * we need to expand dependencies with multiple artifacts into multiple dependencies.
     *
     * @param dependencies Initial dependencies to expand
     * @return dependencies expanded if multiple artifacts are found
     */
    private List<MappedDependency> expandClassifiers(List<MappedDependency> dependencies) {
        List<MappedDependency> expandedDependencies = []

        for (dep in dependencies) {
            if (dep.moduleArtifacts.size() > 1) {
                expandedDependencies += dep.moduleArtifacts.collect { artifact ->
                    // Create a new dependency for this classifier
                    def id = new ResolvedConfigurationIdentifier(artifact.moduleVersion.id, 'compile')
                    new MappedDependency(new StandInResolvedDependency(id, artifact))
                }
            } else {
                expandedDependencies << dep
            }
        }

        // And make sure the list is still unique
        expandedDependencies.unique { d1, d2 ->
            d1.module.id == d2.module.id && d1.artifact.classifier == d2.artifact.classifier ? 0 : 1
        }

        return expandedDependencies
    }

    /**
     * Find all the module descriptors defined in modules XML files.
     *
     * @return flattened list of all module descriptors loaded from modules XML files
     */
    private List<ModuleDescriptor> findModuleDescriptors() {
        // Get the list of initial modules descriptors
        List<ModulesDescriptor> discoveredDescriptors = new ModulesXmlLocator(extension.defaultSlot)
                .discover(Thread.currentThread().contextClassLoader, calcModulesRootDirectories())

        // And flatten that to include all descriptors (when a singe file defines multiple descriptors)
        return discoveredDescriptors*.getDescriptors().flatten() as List<ModuleDescriptor>
    }

    private List<File> calcModulesRootDirectories() {

        final File projectDir = project.projectDir
        boolean emitWarnings = !extension.usingDefaultModules()
        List<String> modules = extension.modules
        List<File> rootDirectories = []

        if (modules != null) {

            for (final String dirPath : modules) {
                // Allow relative or absolute paths
                File dir = new File(dirPath)

                if (dir.isDirectory()) {
                    rootDirectories.add(dir)
                } else {
                    dir = new File(projectDir, dirPath) // Make relative to project root

                    if (dir.isDirectory()) {
                        rootDirectories.add(dir)
                    } else if (emitWarnings) {
                        getLogger().warn("Modules directory '$dirPath' does not exist. Skipping ...")
                    }
                }
            }
        }

        if (extension.verbose) project.getLogger().quiet("Modules Roots: " + rootDirectories)

        return rootDirectories
    }

    @SuppressWarnings(["GroovyUnusedDeclaration", "GroovyAssignabilityCheck"])
    private void dumpDeps() {

        println "Dep: " + compileConfiguration.dependencies
        println "AllDep: " + compileConfiguration.allDependencies
    }

    /**
     * Local implementation of ResolvedDependency and ResolvedModuleVersion so we can avoid being tied so closely
     * to the gradle implementation of DefaultResolvedDependency, which has had it's implementation
     * (including constructors) changed several times recently.
     *
     * @since 1.2
     */
    private class StandInResolvedDependency implements ResolvedDependency {

        private final ResolvedConfigurationIdentifier id
        private final ResolvedModuleVersion resolvedModuleVersion
        private final ResolvedArtifact artifact
        private final String name

        StandInResolvedDependency(ResolvedConfigurationIdentifier id, ResolvedArtifact artifact) {
            this.id = id
            this.artifact = artifact
            this.resolvedModuleVersion = new LocalResolvedModuleVersion(id.getId())

            this.name = String.format("%s:%s:%s", id.getModuleGroup(), id.getModuleName(), id.getModuleVersion())
        }

        @Override
        String getName() {
            return name
        }

        @Override
        String getModuleGroup() {
            return id.getModuleGroup()
        }

        @Override
        String getModuleName() {
            return id.getModuleName()
        }

        @Override
        String getModuleVersion() {
            return id.getModuleVersion()
        }

        @Override
        String getConfiguration() {
            return id.getConfiguration()
        }

        @Override
        ResolvedModuleVersion getModule() {
            return resolvedModuleVersion
        }

        @Override
        Set<ResolvedDependency> getChildren() {
            return Collections.emptySet()
        }

        @Override
        Set<ResolvedDependency> getParents() {
            return Collections.emptySet()
        }

        @Override
        Set<ResolvedArtifact> getModuleArtifacts() {
            return Collections.singleton(artifact)
        }

        @Override
        Set<ResolvedArtifact> getAllModuleArtifacts() {
            return getModuleArtifacts()
        }

        @Override
        Set<ResolvedArtifact> getParentArtifacts(ResolvedDependency parent) {
            return Collections.emptySet()
        }

        @Override
        Set<ResolvedArtifact> getArtifacts(ResolvedDependency parent) {
            return Collections.emptySet()
        }

        @Override
        Set<ResolvedArtifact> getAllArtifacts(ResolvedDependency parent) {
            return Collections.emptySet()
        }
    }

    private static final class LocalResolvedModuleVersion implements ResolvedModuleVersion {
        private final ModuleVersionIdentifier identifier

        LocalResolvedModuleVersion(ModuleVersionIdentifier identifier) {
            this.identifier = identifier
        }

        @Override
        String toString() {
            return identifier.toString()
        }

        ModuleVersionIdentifier getId() {
            return identifier
        }
    }
}
