package gov.raptor.gradle.plugins.jbossmodules

import de.smartics.maven.plugin.jboss.modules.descriptor.ArtifactClusion
import de.smartics.maven.plugin.jboss.modules.descriptor.ModuleDescriptor
import org.gradle.api.Project
import org.gradle.api.artifacts.ResolvedDependency

/**
 * Handle the pruning of the resolved dependencies by applying dependency excludes from the build configuration
 * and "skipped" modules from the modules descriptors.
 */
class DependencyPruner {

    private final Project project
    private final Set<ResolvedDependency> directDependencies
    private final List<ArtifactClusion> dependencyExcludes
    private final List<ModuleDescriptor> skipModules

    DependencyPruner(Project project, List<ArtifactClusion> dependencyExcludes, List<ModuleDescriptor> allModules) {
        this.project = project
        this.dependencyExcludes = dependencyExcludes
        this.skipModules = allModules.grep { it.directives.skip }.collect()

        def compileConfiguration = project.configurations.getByName('compile')
        this.directDependencies = compileConfiguration.resolvedConfiguration.firstLevelModuleDependencies
    }

    /**
     * Scan the resolved dependency DAG and prune any modules that are excluded or skipped.
     *
     * @return Set of dependencies to use for module generation
     */
    Set<ResolvedDependency> getPrunedDependencies() {
        List<ResolvedDependency> prunedDependencies = []

        for (ResolvedDependency dep : directDependencies) {
            doPrune(dep, prunedDependencies)
        }

        project.getLogger().debug("pruned: " + prunedDependencies)

        def unique = prunedDependencies.unique { dep -> dep.getModule().getId() }
        return unique
    }

    /**
     * Recursive prune.  Process a given dependency and recurse on its child dependencies.
     *
     * @param dep Dependency to process
     * @param prunedDependencies List of collected dependencies
     */
    private void doPrune(ResolvedDependency dep, List<ResolvedDependency> prunedDependencies) {
        def artifact = new MappedArtifact(dep)

        if (!dependencyExcluded(artifact)) {
            prunedDependencies << dep

            for (ResolvedDependency childDep : dep.children.grep { it.getConfiguration() == 'runtime' }) {
                doPrune(childDep, prunedDependencies)
            }
        } else {
            project.logger.debug("Pruning dep: " + dep)
        }
    }

    /**
     * Determine if this dependency should be excluded (pruned), which is determined by our dependency excludes and
     * the modules with a "skip" directive.
     *
     * @param dep Dependency to test
     * @return {@code true} if the dependency should be pruned
     */
    private boolean dependencyExcluded(MappedArtifact dep) {
        return dependencyExcludes.grep { it.matches(dep).isMatched() } || skipModules.grep { it.match(dep).isMatched() }
    }
}
