/*
 * NCATS-COMMON
 *
 * Copyright 2019 NIH/NCATS
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package gov.nih.ncats.common.gov.nih.ncats.common.executors;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * A {@link ThreadPoolExecutor} that will actually
 * block inside calls to submit() until the inner blocking queue
 * has room.  This should never use its RejectionPolicy.
 *
 * Created by katzelda on 3/25/16.
 */
public class BlockingSubmitExecutor extends ThreadPoolExecutor{

    /**
     * Create a new instance that will have a fixed number of threads.
     *
     * @param numThreads the number of threads in the thread pool.
     * @param blockingQueueCapacity the capacity of the blocking queue for jobs to be run.  Once the number of waiting
     *                              jobs exceeds this capacity, the submit() calls will block until one of the threads
     *                              completes its currently running task.
     * @return a new ExecutorService; will never be null.
     */
    public static ExecutorService newFixedThreadPool(int numThreads, int blockingQueueCapacity){
        return new BlockingSubmitExecutor(numThreads, numThreads, 0, TimeUnit.MILLISECONDS, blockingQueueCapacity);
    }
    /**
     * Create a new instance that will have a variable number of threads in the pool.
     *
     * @param corePoolSize the minimum number of threads always in the thread pool.
     * @param maximumPoolSize the maximum number of threads that can be in the threadpool.
     * @param keepAliveTime the amount of time an idle thread should be kept alive in the threadpool
     *                      when the pool size is above core.
     * @param unit the time unit of keep alive time.
     * @param blockingQueueCapacity the capacity of the blocking queue for jobs to be run.  Once the number of waiting
     *                              jobs exceeds this capacity, the submit() calls will block until one of the threads
     *                              completes its currently running task.
     *
     * @return a new ExecutorService; will never be null.
     */
    public static ExecutorService newThreadPool(int corePoolSize, int maximumPoolSize, long keepAliveTime, TimeUnit unit, int blockingQueueCapacity) {
        return new BlockingSubmitExecutor(corePoolSize,maximumPoolSize, keepAliveTime, unit, blockingQueueCapacity);
    }
    /**
     * Create a new instance
     * @param corePoolSize the minimum number of threads always in the thread pool.
     * @param maximumPoolSize the maximum number of threads that can be in the threadpool.
     * @param keepAliveTime the amount of time an idle thread should be kept alive in the threadpool
     *                      when the pool size is above core.
     * @param unit the time unit of keep alive time.
     * @param blockingQueueCapacity the capacity of the blocking queue for jobs to be run.  Once the number of waiting
     *                              jobs exceeds this capacity, the submit() calls will block until one of the threads
     *                              completes its currently running task.
     */
    private BlockingSubmitExecutor(int corePoolSize, int maximumPoolSize, long keepAliveTime, TimeUnit unit, int blockingQueueCapacity) {
        super(corePoolSize, maximumPoolSize, keepAliveTime, unit, new BlockingOfferQueue<Runnable>(blockingQueueCapacity));
    }


    private static final class BlockingOfferQueue<E> extends LinkedBlockingDeque<E> {

        public BlockingOfferQueue(int capacity) {
            super(capacity);
        }

        @Override
        public boolean offer(E e) {
            //Executors never call blocking put()
            //but instead calls offer() which doesn't block
            //but immediately returns true/false
            //if the runnable can be put without blocking.
            //If the queue would have blocked, then
            //the Exectuor calls the RejectionPolicy
            //which will either throw Exception and abort
            //or make the runnable run on THIS thread
            //which will make things run much slower
            //because we can't submit any new threads until
            //this (possibly long running) job completes.

            //Therefore, we will change poll
            //to turn it into a put()!
            //which will force the executor to block

            try {
                put(e);
                return true;
            } catch (InterruptedException e1) {
                //we got interrupted
                //propagate up
                Thread.currentThread().interrupt();
            }
            //I don't think this will ever get called unless
            //maybe if we are interrupting?
            //eitherway makes compiler happy...
            return false;
        }
    }
}
