package com.unitypay.billingmodule


import android.app.Activity
import android.util.Log
import com.android.billingclient.api.*
import com.unitypay.billingmodule.billing.Security
import java.io.IOException
import java.util.*

class GooglePayManager(var act: Activity, var mBillingUpdatesListener: BillingUpdatesListener) : PurchasesUpdatedListener {
    val TAG = "GooglePay"

    var mBillingClient: BillingClient = BillingClient.newBuilder(act).setListener(this).build()
    var mBillingClientResponseCode = BillingClient.BillingResponse.SERVICE_DISCONNECTED
    val mPurchases = ArrayList<Purchase>()

    interface BillingUpdatesListener {
        fun onBillingClientSetupFinished()
        fun onConsumeFinished(token: String, @BillingClient.BillingResponse result: Int)
        fun onPurchasesUpdated(purchases: List<Purchase>)
    }


    private val BASE_64_ENCODED_PUBLIC_KEY = "MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAi2vCWSaTqeYYczDu8Jz3tVeNKSjtLEleU6aTvMC/EVI7RHiTuqUOV00QJKj2Bkx86J68MbeAdSXywjkwPFplV+dA2z3lNCp09nNd8Lh5BA0NX7Zl2p5O4jEPD4rGpMu2QAKGHRYixMxTJ7HuMpzvytp0Fme6QYWtFWHukQJ9CNKoKK2kE42eUo62GeIctPHPAYBY6m+vovPaWuiIh25LzPEPjac2qpnDVhdk7AUnGK3Vn4TTH8QujuHWY+u5pN2BaK1ffd7FTb+cOoo7EvM3GTn5iPDuPBicAGFrP7m17V7iywpY0XkJbYD7w4cM+wmZurXJmySYnpQ1UCOda6xCeQIDAQAB"

    var mIsServiceConnected = false

    init {

        startServiceConnection(Runnable {
            mBillingUpdatesListener.onBillingClientSetupFinished()
            // IAB is fully set up. Now, let's get an inventory of stuff we own.
            queryPurchases()
        })
    }


    fun startServiceConnection(executeOnSuccess: Runnable) {

        mBillingClient.startConnection(object : BillingClientStateListener {
            override fun onBillingSetupFinished(@BillingClient.BillingResponse billingResponseCode: Int) {
                if (billingResponseCode == BillingClient.BillingResponse.OK) {
                    mIsServiceConnected = true
                    executeOnSuccess.run()
                }
                mBillingClientResponseCode = billingResponseCode
            }

            override fun onBillingServiceDisconnected() {
                mIsServiceConnected = false
            }
        })
    }


    fun queryPurchases() {
        val queryToExecute = Runnable {
            val time = System.currentTimeMillis()
            val purchasesResult = mBillingClient.queryPurchases(BillingClient.SkuType.INAPP)
            Log.i(TAG, "Querying purchases elapsed time: " + (System.currentTimeMillis() - time)
                    + "ms")
            // If there are subscriptions supported, we add subscription rows as well
            if (areSubscriptionsSupported()) {
                val subscriptionResult = mBillingClient.queryPurchases(BillingClient.SkuType.SUBS)
                Log.i(TAG, "Querying purchases and subscriptions elapsed time: "
                        + (System.currentTimeMillis() - time) + "ms")
                Log.i(TAG, "Querying subscriptions result code: "
                        + subscriptionResult.responseCode
                        + " res: " + subscriptionResult.purchasesList.size)

                if (subscriptionResult.responseCode == BillingClient.BillingResponse.OK) {
                    purchasesResult.purchasesList.addAll(
                            subscriptionResult.purchasesList)
                } else {
                    Log.e(TAG, "Got an error response trying to query subscription purchases")
                }
            } else if (purchasesResult.responseCode == BillingClient.BillingResponse.OK) {
                Log.i(TAG, "Skipped subscription purchases query since they are not supported")
            } else {
                Log.w(TAG, "queryPurchases() got an error response code: " + purchasesResult.responseCode)
            }
            onQueryPurchasesFinished(purchasesResult)
        }

        executeServiceRequest(queryToExecute)
    }

    private fun executeServiceRequest(runnable: Runnable) {
        if (mIsServiceConnected) {
            runnable.run()
        } else {
            // If billing service was disconnected, we try to reconnect 1 time.
            // (feel free to introduce your retry policy here).
            startServiceConnection(runnable)
        }
    }


    fun areSubscriptionsSupported(): Boolean {
        val responseCode = mBillingClient.isFeatureSupported(BillingClient.FeatureType.SUBSCRIPTIONS)
        if (responseCode != BillingClient.BillingResponse.OK) {
            Log.w(TAG, "areSubscriptionsSupported() got an error response: $responseCode")
        }
        return responseCode == BillingClient.BillingResponse.OK
    }

    private fun onQueryPurchasesFinished(result: Purchase.PurchasesResult) {
        // Have we been disposed of in the meantime? If so, or bad result code, then quit
        if (mBillingClient == null || result.responseCode != BillingClient.BillingResponse.OK) {
            Log.w(TAG, "Billing client was null or result code (" + result.responseCode
                    + ") was bad - quitting")
            return
        }

        Log.d(TAG, "Query inventory was successful.")

        // Update the UI and purchases inventory with new list of purchases
        mPurchases.clear()
        onPurchasesUpdated(BillingClient.BillingResponse.OK, result.purchasesList)
    }


    override fun onPurchasesUpdated(responseCode: Int, purchases: MutableList<Purchase>?) {
        if (responseCode == BillingClient.BillingResponse.OK) {
            for (purchas in purchases ?: arrayListOf()) {
                handlePurchase(purchas)
            }
            mBillingUpdatesListener.onPurchasesUpdated(mPurchases)
        } else if (responseCode == BillingClient.BillingResponse.USER_CANCELED) {
            Log.i(TAG, "onPurchasesUpdated() - user cancelled the purchase flow - skipping")
        } else {
            Log.w(TAG, "onPurchasesUpdated() got unknown resultCode: $responseCode")
        }

    }

    private fun handlePurchase(purchase: Purchase) {
        if (!verifyValidSignature(purchase.originalJson, purchase.signature)) {
            Log.i(TAG, "Got a purchase: $purchase; but signature is bad. Skipping...")
            return
        }

        Log.d(TAG, "Got a verified purchase: $purchase")

        mPurchases.add(purchase)
    }


    private fun verifyValidSignature(signedData: String, signature: String): Boolean {
        // Some sanity checks to see if the developer (that's you!) really followed the
        // instructions to run this sample (don't put these checks on your app!)
        if (BASE_64_ENCODED_PUBLIC_KEY.contains("CONSTRUCT_YOUR")) {
            throw RuntimeException("Please update your app's public key at: " + "BASE_64_ENCODED_PUBLIC_KEY")
        }

        try {
            return Security.verifyPurchase(BASE_64_ENCODED_PUBLIC_KEY, signedData, signature)
        } catch (e: IOException) {
            Log.e(TAG, "Got an exception trying to validate a purchase: $e")
            return false
        }

    }


    fun querySkuDetailsAsync(@BillingClient.SkuType itemType: String, skuList: List<String>,
                             listener: SkuDetailsResponseListener) {
        // Creating a runnable from the request to use it inside our connection retry policy below
        val queryRequest = Runnable {
            // Query the purchase async
            val params = SkuDetailsParams.newBuilder()
            params.setSkusList(skuList).setType(itemType)
            mBillingClient.querySkuDetailsAsync(params.build()
            ) { responseCode, skuDetailsList -> listener.onSkuDetailsResponse(responseCode, skuDetailsList) }
        }

        executeServiceRequest(queryRequest)
    }

//    fun consumeAsync(purchaseToken: String) {
//        // If we've already scheduled to consume this token - no action is needed (this could happen
//        // if you received the token when querying purchases inside onReceive() and later from
//        // onActivityResult()
//        if (mTokensToBeConsumed == null) {
//            mTokensToBeConsumed = HashSet<String>()
//        } else if (mTokensToBeConsumed.contains(purchaseToken)) {
//            Log.i(TAG, "Token was already scheduled to be consumed - skipping...")
//            return
//        }
//        mTokensToBeConsumed.add(purchaseToken)
//
//        // Generating Consume Response listener
//        val onConsumeListener = ConsumeResponseListener { responseCode, purchaseToken ->
//            // If billing service was disconnected, we try to reconnect 1 time
//            // (feel free to introduce your retry policy here).
//            mBillingUpdatesListener.onConsumeFinished(purchaseToken, responseCode)
//        }
//
//        // Creating a runnable from the request to use it inside our connection retry policy below
//        val consumeRequest = Runnable {
//            // Consume the purchase async
//            mBillingClient.consumeAsync(purchaseToken, onConsumeListener)
//        }
//
//        executeServiceRequest(consumeRequest)
//    }


}