; Copyright (c) 2013 Gavin "Groovy" Grover. All rights reserved.
; The use and distribution terms for this software are covered by the
; Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
; which can be found in the file epl-v10.html at the root of this distribution.
; By using this software in any fashion, you are agreeing to be bound by
; the terms of this license.
; You must not remove this notice, or any other, from this software.

(ns gggrover.tuples
  (:use clojure.test))

;--------------------------------------------------------------------------------
(defrecord
  ^{:doc "Implementation record for vectuples. vals must be a vector of two or more values."}
  Vectuple [vals])

(defrecord
  ^{:doc
  "Implementation record for listuples. vals must be a lazily-evaluated list
   of two or more values, so at least the first two values must be evaluated."}
  Listuple [vals])

;--------------------------------------------------------------------------------
(defn- vectuple-strip [x]
  (cond
    (= (class x) Vectuple) (:vals x)
    (nil? x) []
    :else [x]))

(defn- listuple-strip [x]
  (cond
    (= (class x) Listuple) (:vals x)
    (nil? x) ()
    :else (list x)))

(defn- struple-strip [x]
  (cond
    (= (class x) String) x
    (nil? x) ""
    :else (str x) ;//must be char
))

;--------------------------------------------------------------------------------
(defn- vectuple-wrap [^:clojure.lang.PersistentList x]
  (cond
    (= 0 (count x)) nil
    (= 1 (count x)) (first x)
    :else (->Vectuple (vec x))))

(defn- listuple-wrap [^:clojure.lang.PersistentList x]
  (cond
    (nil? (first x)) nil
    (nil? (second x)) (first x)
    :else (->Listuple x)))

(defn- struple-wrap [^:java.lang.String x]
  (cond
    (= 0 (count x)) nil
    (= 1 (count x)) (get x 0)
    :else x))

;--------------------------------------------------------------------------------
(defn vectuple
  "Creates a vectuple from given objects and Vectuple records. A vectuple is
   either a nil, a single object, or a Vectuple record of two or more
   objects, none of which are nil or themselves a Vectuple record."
  [& x]
    (vectuple-wrap (apply concat (map vectuple-strip x))))

(defn listuple
  "Creates a listuple from given objects and Listuple records. A listuple is
   either a nil, a single object, or a Listuple record of two or more
   objects, none of which are nil or themselves a Listuple record."
  [& x]
    (listuple-wrap (apply concat (map listuple-strip x))))

(defn struple
  "Creates a struple from given chars and Strings. A struple is
   either a nil, a single char, or a String of two or more chars."
  [& x]
    (struple-wrap (apply str (map struple-strip x))))

;--------------------------------------------------------------------------------
(defn vectuple-count
  "Return size of vectuple."
  [x]
    (cond
      (nil? x) 0
      (not= (class x) Vectuple) 1
      :else (count (:vals x))))

(defn listuple-count
  "Return size of listuple."
  [x]
    (cond
      (nil? x) 0
      (not= (class x) Listuple) 1
      :else (count (:vals x))))

(defn struple-count
  "Return size of struple."
  [x]
    (cond
      (nil? x) 0
      (char? x) 1
      :else (count x)))

;--------------------------------------------------------------------------------
(defn vectuple-get
  "Get 1-based index from vectuple, return vectuple if index is 0,
   or nil if vectuple not large enough."
  [x n]
    (cond
      (< n 0)                (throw (new Exception "tuple index must be 0 or greater"))
      (= n 0)                x
      (= (class x) Vectuple) (nth (:vals x) (dec n) nil)
      (= n 1)                x
      :else                  nil))

(defn struple-get
  "Get 1-based index from struple, return struple if index is 0,
   or nil if struple not large enough."
  [x n]
    (cond
      (< n 0)              (throw (new Exception "struple index must be 0 or greater"))
      (= n 0)              x
      (string? x)          (nth x (dec n) nil)
      (= n 1)              x
      :else                nil))

;--------------------------------------------------------------------------------
(defn vectuple-take
  "Take first n elements of vectuple as vector of size n, ignoring extra elements.
   If not enough items, append nils."
  ([x n]
    (if (< n 0)
      (throw (new Exception "tuple index must be 0 or greater"))
      (if (and (nil? x) (= n 0)) []
        (let [y (vectuple-strip x)]
          (into
            (vec (take n y))
            (repeat (- n (count y)) nil)))))))

(defn struple-take
  "Take first n elements of struple as string of size n, ignoring extra elements.
   If not enough items, truncate result."
  ([x n]
    (if (< n 0)
      (throw (new Exception "struple index must be 0 or greater"))
      (if (and (nil? x) (= n 0)) ""
        (let [y (struple-strip x)]
                (subs y 0 (min n (count y))))))))

;--------------------------------------------------------------------------------
(defn vectuple-dissolve
  "Convert vectuple to vector of size n, with last element a vectuple containing
   remaining items. If not enough items, append nils. If size 0 is given,
   return vector with all elements of vectuple, or empty list if vectuple is nil."
  ([x] (vectuple-dissolve x 0))
  ([x n]
    (if (< n 0)
      (throw (new Exception "tuple index must be 0 or greater"))
      (if (and (nil? x) (= n 0)) []
        (let [y (vectuple-strip x)
              c (if (= n 0) (count y) n)]
          (apply conj
            (vec (take (dec c) y))
            (apply vectuple (drop (dec c) y))
            (repeat (- (dec c) (count y)) nil)))))))

;--------------------------------------------------------------------------------
(defn vectuple-prepend
  "Prepend to the last element, which must be a vector, the previous elements
   as a vectuple."
  [& x]
    (let [v (last x)
          init (apply vectuple (butlast x))]
      (if (empty? v)
        [init]
        (apply conj [init] v))))

;--------------------------------------------------------------------------------
(defn vectuple-invoke
  "For each element in vectuple x, invokes f with element as parameter."
  ([f] (f))
  ([f x]
    (cond
      (= (class x) Vectuple) (vectuple-wrap (map f (vectuple-strip x)))
      (nil? x) (f)
      :else (f x))))

;--------------------------------------------------------------------------------
