package foundation.stack.docker;

import com.google.common.io.ByteProcessor;
import com.google.common.io.ByteStreams;
import com.google.common.io.Files;
import org.kohsuke.github.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.BiConsumer;
import java.util.function.Supplier;

/**
 * @author Ravi Chodavarapu (rchodava@gmail.com)
 */
public class DockerDownloader {
    private static final Logger logger = LoggerFactory.getLogger(DockerDownloader.class);

    private static final Map<Os, String> osToInstallerExtension = new HashMap<>(2);

    private static final String DOCKER_USER = "docker";
    private static final String TOOLBOX_REPOSITORY = "toolbox";

    private static final String PKG_EXTENSION = "pkg";
    private static final String EXE_EXTENSION = "exe";
    private static final String DOCKER_INSTALL_SCRIPT_URL = "https://get.docker.com/";

    static {
        osToInstallerExtension.put(Os.MacOS, PKG_EXTENSION);
        osToInstallerExtension.put(Os.Windows, EXE_EXTENSION);
    }

    private static GHAsset getLatestReleaseAsset(Os os) throws IOException {
        GitHub github = GitHub.connectAnonymously();
        GHUser dockerUser = github.getUser(DOCKER_USER);
        if (dockerUser != null) {
            GHRepository toolboxRepository = dockerUser.getRepository(TOOLBOX_REPOSITORY);
            PagedIterable<GHRelease> releases = toolboxRepository.listReleases();

            for (GHRelease release : releases) {
                if (!release.isDraft() && !release.isPrerelease()) {
                    logger.debug("Downloading latest version of Docker Toolbox - {}", release.getName());

                    List<GHAsset> assets = release.getAssets();
                    if (assets != null) {
                        String extension = osToInstallerExtension.get(os);
                        if (extension != null) {
                            for (GHAsset asset : assets) {
                                String downloadUrl = asset.getBrowserDownloadUrl();
                                if (downloadUrl != null && downloadUrl.toLowerCase().endsWith(extension)) {
                                    logger.debug("Downloading Docker Toolbox from {} ({} MB)", downloadUrl, asset.getSize() / (1024 * 1024));
                                    return asset;
                                }
                            }
                        }
                    }
                }
            }
        }

        logger.debug("Could not download installer for operating system {}", os);
        return null;
    }

    private static File downloadDockerToolbox(BiConsumer<Integer, Integer> progressUpdater, Supplier<Boolean> isCancelled) throws IOException {
        GHAsset asset = getLatestReleaseAsset(Os.getSystemOs());
        if (asset != null) {
            String downloadUrl = asset.getBrowserDownloadUrl();

            int totalSize = (int) asset.getSize();
            return downloadFileAtUrl(progressUpdater, isCancelled, downloadUrl, totalSize);
        }

        return null;
    }

    private static File downloadFileAtUrl(final BiConsumer<Integer, Integer> progressUpdater, final Supplier<Boolean> isCancelled,
                                          String downloadUrl, final int totalSize) throws IOException {
        File temporaryDirectory = Files.createTempDir();

        if (!temporaryDirectory.exists()) {
            temporaryDirectory.mkdirs();
        }

        String fileName = downloadUrl.substring(downloadUrl.lastIndexOf('/'));
        if (fileName == null || fileName.isEmpty() || "/".equals(fileName)) {
            fileName = "install.sh";
        }

        File downloadedFile = new File(temporaryDirectory, fileName);
        logger.debug("Downloading to {}", downloadedFile);

        try (InputStream downloadStream = new URL(downloadUrl).openStream();
             FileOutputStream outputStream = new FileOutputStream(downloadedFile)) {
            return ByteStreams.readBytes(downloadStream, new ByteProcessor<File>() {
                int workDone = 0;

                @Override
                public boolean processBytes(byte[] buffer, int off, int len) throws IOException {
                    outputStream.write(buffer, off, len);
                    workDone += len;
                    progressUpdater.accept(workDone, totalSize);
                    return !isCancelled.get();
                }

                @Override
                public File getResult() {
                    if (!isCancelled.get()) {
                        logger.debug("Finished downloading to {}", downloadedFile);
                    } else {
                        logger.debug("Download to {} was cancelled", downloadedFile);
                    }

                    return downloadedFile;
                }
            });
        }
    }

    public static File downloadDocker(BiConsumer<Integer, Integer> progressUpdater, Supplier<Boolean> isCancelled) throws IOException {
        progressUpdater.accept(0, 0);

        if (Os.getSystemOs() == Os.Windows || Os.getSystemOs() == Os.MacOS) {
            return downloadDockerToolbox(progressUpdater, isCancelled);
        } else {
            return downloadFileAtUrl(progressUpdater, isCancelled, DOCKER_INSTALL_SCRIPT_URL, -1);
        }
    }
}
