package foundation.stack.docker;

import com.github.dockerjava.core.DockerClientBuilder;
import com.github.dockerjava.core.DockerClientConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.util.Arrays;
import java.util.List;

/**
 * @author Ravi Chodavarapu (rchodava@gmail.com)
 */
public class DockerMachineClient {
    private static final Logger logger = LoggerFactory.getLogger(DockerMachineClient.class);

    private static String getMachineExecutable() throws Exception {
        switch (Os.getSystemOs()){
            case Windows: return DockerDetector.getDockerInstallPathForWindows() + File.separatorChar + "docker-machine.exe";
            default: return DockerDetector.getDockerInstallPathForUnix("docker-machine");
        }
    }

    private final String machineExecutable;

    public DockerMachineClient() throws Exception {
        this.machineExecutable = getMachineExecutable();
    }

    public boolean isMachineCreated(String machine) throws Exception {
        logger.debug("Checking if Docker machine {} has been created", machine);

        List<String> result = ProcessRunner.runProcess(machineExecutable, "status", machine);
        if (result != null && result.size() > 0 && ("Running".equals(result.get(0)) || "Stopped".equals(result.get(0)))) {
            return true;
        }

        return false;
    }

    public boolean isMachineRunning(String machine) throws Exception {
        logger.debug("Checking if Docker machine {} is running", machine);

        List<String> result = ProcessRunner.runProcess(machineExecutable, "status", machine);
        if (result != null && result.size() > 0 && "Running".equals(result.get(0))) {
            return true;
        }

        return false;
    }

    public void createMachine(String machine) throws Exception {
        logger.debug("Creating Docker machine {} (this can take a few long minutes)...", machine);

        List<String> result = ProcessRunner.runProcess(machineExecutable, "create", "--driver", "virtualbox", machine);
        if (result == null || !isMachineCreated(machine)) {
            throw new IllegalStateException("Failed to create a new Docker machine!");
        }
    }

    public void createMachineIfNecessary(String machine) throws Exception {
        if (!isMachineCreated(machine)) {
            createMachine(machine);
        }
    }

    public void startMachine(String machine) throws Exception {
        logger.debug("Starting Docker machine {}", machine);

        List<String> result = ProcessRunner.runProcess(machineExecutable, "start", machine);
        if (result == null || !isMachineRunning(machine)) {
            throw new IllegalStateException("Failed to start Docker machine!");
        }
    }

    public void startDockerMachineIfNecessary(String machine) throws Exception {
        if (!isMachineRunning(machine)) {
            startMachine(machine);
        }
    }

    public void setupMachineIfNecessary(String machine) throws Exception {
        createMachineIfNecessary(machine);
        startDockerMachineIfNecessary(machine);
    }


    public DockerClient connect(String machine) throws Exception {
        setupMachineIfNecessary(machine);

        List<String> result = ProcessRunner.runProcess(machineExecutable, "config", machine);
        if (result != null && result.size() > 0) {
            ConfigParams configParams;
            if (result.size() == 1) {
                configParams = inferConfigParams(Arrays.asList(result.get(0).split(" ")));
            }
            else {
                configParams = inferConfigParams(result);
            }

            DockerClientConfig config = DockerClientConfig.createDefaultConfigBuilder()
                    .withDockerTlsVerify(configParams.isTlsVerify())
                    .withDockerCertPath(configParams.getClientCertificatePath())
                    .withDockerHost(configParams.getHost())
                    .build();

            return new DelegatingDockerClient(DockerClientBuilder.getInstance(config).build(), configParams.getHost());
        }

        return null;
    }

    private static ConfigParams inferConfigParams(List<String> configCommandResult) {

        ConfigParams configParams = new ConfigParams();
        for (String config : configCommandResult) {
            String[] params = config.split(" ");
            for(String param : params) {
                if ("--tlsverify".equals(param.trim())) {
                    configParams.setTlsVerify(true);
                } else if (param.startsWith("--tlscert=")) {
                    String clientCertificatePath = param.substring(10);
                    clientCertificatePath = clientCertificatePath.substring(1, clientCertificatePath.length() - 1);
                    int dirSeparator = clientCertificatePath.lastIndexOf(File.separatorChar);
                    configParams.setClientCertificatePath(clientCertificatePath.substring(0, dirSeparator));
                } else if (param.startsWith("-H=")) {
                    configParams.setHost(param.substring(3));
                }
            }
        }
        return configParams;
    }

    private static class ConfigParams {
        private boolean tlsVerify;
        private String clientCertificatePath;
        private String host;

        public boolean isTlsVerify() {
            return tlsVerify;
        }

        public void setTlsVerify(boolean tlsVerify) {
            this.tlsVerify = tlsVerify;
        }

        public String getClientCertificatePath() {
            return clientCertificatePath;
        }

        public void setClientCertificatePath(String clientCertificatePath) {
            this.clientCertificatePath = clientCertificatePath;
        }

        public String getHost() {
            return host;
        }

        public void setHost(String host) {
            this.host = host;
        }
    }
}
