package foundation.stack.docker;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.util.List;

/**
 * @author Ravi Chodavarapu (rchodava@gmail.com)
 */
public class DockerInstaller {
    private static final Logger logger = LoggerFactory.getLogger(DockerInstaller.class);

    public static int installDocker(File file) throws IOException {
        if (file != null) {
            logger.debug("Launching Docker installer at {}", file);
            int result = executeInstaller(file);
            postInstallWork();
            return result;
        }

        return -1;
    }

    private static int executeInstaller(File file) throws IOException {
        int result;
        try {
            result = executeInstallerCommand(file);
        } catch (InterruptedException e) {
            logger.debug("Error waiting for Docker installer", e);
            return -1;
        }

        logger.debug("Installer command execution finished with result {}", result);
        return result;
    }

    private static void postInstallWork() throws IOException {
        if (Os.getSystemOs() == Os.Linux) {
            boolean userBelongsToDockerGroup = false;

            try {
                List<String> output = ProcessRunner.runProcess("groups");
                if (output != null && output.size() > 0) {
                    String groupsLine = output.get(0);
                    String[] groups = groupsLine.split(" ");
                    if (groups.length > 0) {
                        for (String group : groups) {
                            if ("docker".equalsIgnoreCase(group)) {
                                userBelongsToDockerGroup = true;
                            }
                        }
                    }
                }
            } catch (InterruptedException e) {
            }

            if (!userBelongsToDockerGroup) {
                logger.debug("Current user does not belong to docker group - trying to add the user now");
                try {
                    ProcessRunner.runProcess("sudo", "usermod", "-aG", "docker", System.getProperty("user.name"));
                } catch (IOException | InterruptedException e) {
                    logger.debug("Failed to add user to docker group");
                }
            }
        }
    }

    private static int executeInstallerCommand(File file) throws IOException, InterruptedException {
        switch (Os.getSystemOs()) {
            case Windows:
                return ProcessRunner.runProcessAndGetResult(file.getAbsolutePath());
            case MacOS:
                return ProcessRunner.runProcessAndGetResult("open", "-W", file.getAbsolutePath());
            case Linux:
                return ProcessRunner.runProcessAndGetResult("sh", file.getAbsolutePath());
        }

        throw new IllegalStateException("Could not figure platform when trying to run installer");
    }
}
