package foundation.stack.docker;

import kamon.sigar.SigarProvisioner;
import org.hyperic.sigar.win32.RegistryKey;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.List;

/**
 * @author Ravi Chodavarapu (rchodava@gmail.com)
 */
public class DockerDetector {
    private static final Logger logger = LoggerFactory.getLogger(DockerDetector.class);

    public static boolean isDockerInstalled() throws Exception {
        logger.debug("Checking whether Docker is installed");
        switch (Os.getSystemOs()) {
            case Windows: return isDockerInstalledWindows();
            default: return isDockerInstalledUnix();
        }
    }

    private static boolean isDockerInstalledUnix() throws IOException, InterruptedException {
        String result = getDockerInstallPathForUnix("docker");
        return result != null && !result.isEmpty();
    }

    public static String getDockerInstallPathForUnix(String binary) throws IOException, InterruptedException {
        List<String> result = ProcessRunner.runProcess("which", binary);
        if (result != null && result.size() > 0 && !result.get(0).isEmpty()) {
            // TODO: more comprehensive checks?
            logger.debug("Found docker installed at {}", result.get(0));
            return result.get(0);
        }
        logger.debug("Couldn't find Docker on the PATH");
        return null;
    }

    private static boolean provisionSigar(int alternateLocationAttempt) {
        File attemptLocation = new File(SigarProvisioner.discoverLocation(null), "docker-bootstrap-" + alternateLocationAttempt);
        try {
            SigarProvisioner.provision(attemptLocation);
            return true;
        } catch (Exception e) {
            if (alternateLocationAttempt < 25) {
                return provisionSigar(alternateLocationAttempt + 1);
            } else {
                return false;
            }
        }
    }

    private static void provisionSigarIfNecessary() throws Exception {
        if (!SigarProvisioner.isNativeLoaded()) {
            provisionSigar(0);
        }
    }

    public static String getDockerInstallPathForWindows() throws Exception {
        provisionSigarIfNecessary();

        RegistryKey environment = null;
        String toolboxPath = null;
        try {
            environment = RegistryKey.CurrentUser.openSubKey("Environment");
            if (environment != null) {
                toolboxPath = environment.getStringValue("DOCKER_TOOLBOX_INSTALL_PATH");
                if (toolboxPath != null) {
                    toolboxPath = toolboxPath.trim();
                }
            }
        } finally {
            if (environment != null) {
                environment.close();
            }
        }

        return toolboxPath;
    }

    private static boolean isDockerInstalledWindows() throws Exception {
        provisionSigarIfNecessary();

        String toolboxPath = getDockerInstallPathForWindows();

        if (toolboxPath != null) {
            logger.debug("Found a registered Docker Toolbox installation at {}", toolboxPath);
        }

        File toolboxInstallationFile = new File(toolboxPath);
        if (toolboxInstallationFile.exists() && toolboxInstallationFile.isDirectory()) {
            logger.debug("Docker Toolbox installation at {} is valid", toolboxPath);
            return true;
        } else {
            logger.debug("Docker Toolbox at {} seems to have been uninstalled", toolboxPath);
        }

        return false;
    }
}
