package foundation.stack.docker;

import javafx.application.Application;
import javafx.concurrent.Task;
import javafx.event.ActionEvent;
import javafx.geometry.Insets;
import javafx.geometry.Pos;
import javafx.scene.Scene;
import javafx.scene.control.*;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.layout.*;
import javafx.scene.paint.Color;
import javafx.scene.text.Font;
import javafx.scene.text.FontWeight;
import javafx.scene.text.Text;
import javafx.stage.Modality;
import javafx.stage.Stage;
import javafx.stage.StageStyle;
import javafx.stage.WindowEvent;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.util.Optional;

/**
 * @author Ravi Chodavarapu (rchodava@gmail.com)
 */
public class BootstrapWizard extends Application {
    private static final Logger logger = LoggerFactory.getLogger(BootstrapWizard.class);

    @Override
    public void start(Stage primaryStage) {
        GridPane body = createBody();
        createAndAddLogo(body);
        createAndAddWelcomeText(body);
        createAndAddMessageText(body);
        Button continueButton = createAndAddContinueButton(body);

        continueButton.setOnAction(event -> handleContinue(primaryStage, continueButton, event));

        Scene scene = new Scene(body);
        primaryStage.setScene(scene);
        primaryStage.setTitle(Messages.translate("title"));
        primaryStage.setWidth(680);
        primaryStage.setResizable(false);
        primaryStage.show();

        primaryStage.setOnCloseRequest(this::handleClose);
    }

    private void attemptInstall(File downloadedFile, Button continueButton) {
        Alert alert = new Alert(Alert.AlertType.INFORMATION);
        alert.setTitle(Messages.translate("launchingInstallerTitle"));
        alert.setHeaderText(Messages.translate("launchingInstallerMessage"));
        alert.getButtonTypes().setAll();

        alert.setOnCloseRequest(e -> e.consume());
        alert.show();

        int result = -1;
        try {
            result = DockerInstaller.installDocker(downloadedFile);
        } catch (IOException e) {
        }

        alert.setOnCloseRequest(null);
        alert.setResult(ButtonType.CANCEL);
        alert.close();

        try {
            if (result == 2) {
                logger.debug("Docker install was cancelled");
            } else if (result != 0 || !DockerDetector.isDockerInstalled()) {
                handleInstallError();
            }
        } catch (Exception e) {
            logger.debug("Error while trying to determine if Docker installation was successful");
        }

        continueButton.setDisable(false);
    }

    private void attemptDownload(Stage primaryStage, Button continueButton) {
        ProgressDialog progressDialog = new ProgressDialog();

        Task<File> task = new Task<File>() {
            @Override
            public File call() throws IOException {
                return DockerDownloader.downloadDocker(
                        (workDone, totalWork) -> updateProgress(workDone, totalWork),
                        () -> isCancelled());
            }
        };

        progressDialog.activateProgressBar(task);

        task.setOnSucceeded(__ -> {
            progressDialog.getDialogStage().close();

            logger.debug("Proceeding to install");
            attemptInstall(task.getValue(), continueButton);
        });

        task.setOnFailed(__ -> {
            progressDialog.getDialogStage().close();
            continueButton.setDisable(false);

            logger.debug("Download failed with exception", task.getException());
            handleDownloadError(primaryStage, continueButton);
        });

        progressDialog.getDialogStage().setOnCloseRequest(request -> {
            handleClose(request);
            if (!request.isConsumed()) {
                task.cancel();
                continueButton.setDisable(false);
            }
        });

        continueButton.setDisable(true);
        progressDialog.getDialogStage().show();

        Thread thread = new Thread(task);
        thread.start();
    }

    private void handleContinue(Stage primaryStage, Button continueButton, ActionEvent _) {
        attemptDownload(primaryStage, continueButton);
    }

    private Button createAndAddContinueButton(GridPane body) {
        Button continueButton = new Button(Messages.translate("continue"));
        continueButton.setFont(Font.font(Font.getDefault().getName(), FontWeight.BOLD, 18));
        HBox buttonContainer = new HBox(10);
        buttonContainer.setAlignment(Pos.BOTTOM_RIGHT);
        buttonContainer.getChildren().add(continueButton);
        body.add(buttonContainer, 1, 4);
        return continueButton;
    }

    private GridPane createBody() {
        GridPane grid = new GridPane();
        grid.setBackground(new Background(new BackgroundFill(Color.WHITE, CornerRadii.EMPTY, Insets.EMPTY)));
        grid.setAlignment(Pos.CENTER);
        grid.setHgap(10);
        grid.setVgap(10);
        grid.setPadding(new Insets(25, 25, 25, 25));
        return grid;
    }

    private void createAndAddMessageText(GridPane grid) {
        Text messageText = new Text(Messages.translate("message"));
        messageText.setFont(Font.font(Font.getDefault().getName(), FontWeight.NORMAL, 16));
        messageText.setWrappingWidth(600);
        grid.add(messageText, 0, 2, 2, 1);
    }

    private void createAndAddWelcomeText(GridPane grid) {
        Text welcomeText = new Text(Messages.translate("welcome"));
        welcomeText.setFont(Font.font(Font.getDefault().getName(), FontWeight.BOLD, 18));
        grid.add(welcomeText, 0, 1, 1, 1);
    }

    private void createAndAddLogo(GridPane grid) {
        Image logo = new Image("logo.png");
        ImageView logoView = new ImageView();
        logoView.setImage(logo);
        logoView.setFitWidth(350);
        logoView.setPreserveRatio(true);
        logoView.setSmooth(true);
        logoView.setCache(true);
        grid.add(logoView, 0, 0, 1, 1);
    }

    private void handleDownloadError(Stage primaryStage, Button continueButton) {
        Alert alert = new Alert(Alert.AlertType.ERROR);
        alert.setTitle(Messages.translate("downloadErrorTitle"));
        alert.setHeaderText(Messages.translate("downloadErrorMessage"));
        alert.setContentText(Messages.translate("downloadErrorConfirmation"));
        alert.getButtonTypes().setAll(ButtonType.YES, ButtonType.NO);

        Optional<ButtonType> result = alert.showAndWait();
        if (result.get() == ButtonType.NO) {
            // Don't attempt retry
        } else {
            attemptDownload(primaryStage, continueButton);
        }
    }

    private void handleInstallError() {
        Alert alert = new Alert(Alert.AlertType.ERROR);
        alert.setTitle(Messages.translate("installErrorTitle"));
        alert.setHeaderText(Messages.translate("installErrorMessage"));
        alert.getButtonTypes().setAll(ButtonType.OK);

        alert.showAndWait();
    }

    private void handleClose(WindowEvent e) {
        Alert alert = new Alert(Alert.AlertType.CONFIRMATION);
        alert.setTitle(Messages.translate("cancelTitle"));
        alert.setHeaderText(Messages.translate("cancelMessage"));
        alert.setContentText(Messages.translate("cancelConfirmation"));
        alert.getButtonTypes().setAll(ButtonType.YES, ButtonType.NO);

        Optional<ButtonType> result = alert.showAndWait();
        if (result.get() == ButtonType.NO) {
            // Cancel
            e.consume();
        }
    }

    public class ProgressDialog {
        private final Stage dialogStage;
        private final ProgressBar progressBar = new ProgressBar();
        private final Label progressMessage = new Label();

        public ProgressDialog() {
            dialogStage = new Stage();
            dialogStage.setTitle(Messages.translate("progressTitle"));
            dialogStage.initStyle(StageStyle.UTILITY);
            dialogStage.setResizable(false);
            dialogStage.initModality(Modality.APPLICATION_MODAL);

            Label message = new Label();
            message.setText(Messages.translate("progressMessage"));

            progressBar.setProgress(-1f);
            progressMessage.setText("");

            HBox progressLayout = new HBox();
            progressLayout.setSpacing(10);
            progressLayout.getChildren().add(progressBar);
            progressLayout.getChildren().add(progressMessage);
            HBox.setHgrow(progressBar, Priority.ALWAYS);

            VBox layout = new VBox();
            layout.setPadding(new Insets(15, 15, 15, 15));
            layout.setSpacing(10);
            layout.getChildren().add(message);
            layout.getChildren().add(progressLayout);

            progressBar.prefWidthProperty().bind(
                    progressLayout.widthProperty()
                            .subtract(progressMessage.widthProperty())
                            .subtract(10));

            Scene scene = new Scene(layout);
            dialogStage.setScene(scene);
        }

        public void activateProgressBar(final Task<?> task)  {
            progressBar.progressProperty().bind(task.progressProperty());
            progressMessage.textProperty().bind(
                    task.workDoneProperty().divide(1024 * 1024).asString("%.1f")
                            .concat("/")
                            .concat(task.totalWorkProperty().divide(1024 * 1024).asString("%.1f"))
                            .concat(" MB"));
            dialogStage.show();
        }

        public Stage getDialogStage() {
            return dialogStage;
        }
    }

    public static void launch() {
        launch(BootstrapWizard.class);
    }
}