package foundation.stack.docker;

import com.github.dockerjava.api.model.Container;
import com.github.dockerjava.core.DockerClientBuilder;
import com.github.dockerjava.core.DockerClientConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.InetAddress;
import java.util.List;

/**
 * @author Ravi Chodavarapu (rchodava@gmail.com)
 */
public class Bootstrap {
    private static final Logger logger = LoggerFactory.getLogger(Bootstrap.class);

    /**
     * Main entry-point which will bootstrap Docker and connect to the Docker daemon.
     *
     * @param machine On Mac and Windows, where the Docker container is on a virtual host, the name of the machine to
     *                setup with docker-machine and the machine which has the daemon to connect to.
     * @param bypassInstallation whether or not docker installation should be attempted
     *
     * @return A client connected to the Docker daemon.
     */
    public static DockerClient bootstrapAndConnect(String machine, boolean bypassInstallation) throws Exception {
        if (!bypassInstallation) {
            if (!DockerDetector.isDockerInstalled()) {
                boolean javaFxPresent = true;
                try {
                    Bootstrap.class.getClassLoader().loadClass("javafx.application.Application");
                } catch (ClassNotFoundException e) {
                    javaFxPresent = false;
                }

                if ((Os.getSystemOs() == Os.Windows || Os.getSystemOs() == Os.MacOS) && javaFxPresent) {
                    BootstrapWizard.launch();
                }
                else {
                    logger.debug("The environment does not support graphics, switching to headless");
                    BootstrapCommandLineWizard.launch();
                }
            }
            else {
                logger.debug("Docker is already installed");
            }
        }
        else {
            logger.debug("Requested to bypass docker installation");
        }

        if (DockerDetector.isDockerInstalled()) {
            if (Os.getSystemOs() == Os.Windows || Os.getSystemOs() == Os.MacOS) {
                DockerMachineClient machineClient = new DockerMachineClient();
                return machineClient.connect(machine);
            } else {
                return new DelegatingDockerClient(
                        DockerClientBuilder.getInstance(DockerClientConfig.createDefaultConfigBuilder()).build(),
                        InetAddress.getLocalHost().getHostAddress());
            }
        }
        else {
            logger.debug("Docker is not installed, assuming java client is meant to be used..");
            logger.debug("Will be using defaults, use environment variables to override them if they don't suit. ");
            return new DelegatingDockerClient(
                    DockerClientBuilder.getInstance(DockerClientConfig.createDefaultConfigBuilder()).build(),
                    InetAddress.getLocalHost().getHostAddress());
        }
    }

    /**
     * Main entry-point which will bootstrap Docker and connect to the Docker daemon, always attempting to perform a
     * docker installation first.
     *
     * @param machine On Mac and Windows, where the Docker container is on a virtual host, the name of the machine to
     *                setup with docker-machine and the machine which has the daemon to connect to.
     *
     * @return A client connected to the Docker daemon.
     */
    public static DockerClient bootstrapAndConnect(String machine) throws Exception {
        return bootstrapAndConnect(machine, false);
    }

    public static void main(String[] a) throws Exception {
        List<Container> containers = bootstrapAndConnect("test", false).listContainersCmd().exec();
        for (Container container : containers) {
            System.out.println(container.getId());
        }
    }
}
