package fluent.validation.collections;

import java.time.Duration;
import java.util.Queue;
import java.util.function.Function;
import java.util.function.Supplier;

import static java.util.Objects.requireNonNull;

public final class ValidationCollections {


    public static <D> RepeatingIterable<D> repeatingIterable(Supplier<? extends D> supplier, int maxAttempts, Duration delay) {
        requireNonNull(delay, "Delay must not be null.");
        return repeatingIterable(supplier, maxAttempts, () -> {
            try {
                Thread.sleep(delay.toMillis());
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new RuntimeException(e);
            }
        });
    }

    public static <D> RepeatingIterable<D> repeatingIterable(Supplier<? extends D> supplier, int maxAttempts, Runnable update) {
        return new RepeatingIterable<>(supplier, maxAttempts, update);
    }

    public static <D> ValidationQueue<D> validationQueue(Duration timeout, ValidationQueue.Mode mode) {
        return new ValidationQueue<>(timeout, mode);
    }

    public static <D> ValidationQueue<D> validationQueue(Duration timeout) {
        return validationQueue(timeout, ValidationQueue.Mode.KEEP_ALL);
    }

    public static <D> ValidationQueue<D> removingValidationQueue(Duration timeout) {
        return validationQueue(timeout, ValidationQueue.Mode.REMOVE_ON_NEXT);
    }

    public static <K, D> ValidationCategories<K, D> categories(Function<? super D, ? extends K> categoryFunction, Duration timeout, ValidationQueue.Mode mode) {
        return new ValidationCategories<>(categoryFunction, timeout, mode);
    }

    /**
     * Create a "view" on the original queue with different timeout applied on the blocking operations.
     *
     * Since we only implement the basic Queue interface, there are no methods with timeout parameter. So the timeout
     * is always hidden and set for the whole queue. But there might be use cases, that we want to have custom timeout
     * in various situations. For that purpose we have this "view".
     *
     * You can think of the resulted queue really as being the original one (and it in fact is).
     *
     * @param queue Original queue.
     * @param timeout New timeout to apply in this view.
     * @param <E> Type of the elements in the queue.
     * @return Queue view with custom timeout.
     */
    public static <E> Queue<E> view(Queue<E> queue, Duration timeout) {
        return queue instanceof ValidationQueue ? ((ValidationQueue<E>) queue).view(timeout) : queue;
    }

}
