/*
 * Copyright © 2018 Ondrej Fischer. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that
 * the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
 *    following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other materials provided with the distribution.
 *
 * 3. The name of the author may not be used to endorse or promote products derived from this software without
 *     specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY [LICENSOR] "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT
 * NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package fluent.validation.collections;

import java.time.Duration;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;

public final class ValidationCategories<K, T> implements Consumer<T>, Function<K, ValidationQueue<T>> {

    private final Search<T> search = new Search<>();
    private final Map<K, ValidationQueue<T>> queues = new ConcurrentHashMap<>();
    private final List<T> timeWindowBuffer = new LinkedList<>();
    private final Function<? super T, ? extends K> categoryFunction;
    private final Duration timeout;
    private final ValidationQueue.Mode mode;

    public ValidationCategories(Function<? super T, ? extends K> categoryFunction, Duration timeout, ValidationQueue.Mode mode) {
        this.categoryFunction = categoryFunction;
        this.timeout = timeout;
        this.mode = mode;
    }

    @Override
    public void accept(T data) {
        K categoryKey = categoryFunction.apply(data);
        if(search.accept(data)) {
            get(categoryKey).add(data);
        } else if(queues.containsKey(categoryKey)) {
            queues.computeIfAbsent(categoryKey, this).add(data);
        } else {
            timeWindowBuffer.add(data);
        }
    }

    public ValidationQueue<T> get(K key) {
        ValidationQueue<T> queue = queues.computeIfAbsent(key, this);
        Iterator<T> iterator = timeWindowBuffer.iterator();
        while(iterator.hasNext()) {
            T data = iterator.next();
            if(key.equals(categoryFunction.apply(data))) {
                iterator.remove();
                queue.add(data);
            }
        }
        return queue;
    }

    public CompletableFuture<T> search(Predicate<? super T> predicate) {
        for(T data : timeWindowBuffer) {
            if(predicate.test(data)) {
                get(categoryFunction.apply(data));
                CompletableFuture<T> future = new CompletableFuture<>();
                future.complete(data);
                return future;
            }
        }
        return search.search(predicate);
    }

    @Override
    public ValidationQueue<T> apply(K k) {
        return new ValidationQueue<>(timeout, mode);
    }

}
