/*
 * BSD 2-Clause License
 *
 * Copyright (c) 2020, Ondrej Fischer
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

package foundation.rpg.generator;

import foundation.rpg.automata.LrAction;
import foundation.rpg.automata.LrItem;
import foundation.rpg.automata.LrItemSet;
import foundation.rpg.automata.LrParserAutomata;
import foundation.rpg.grammar.Grammar;
import foundation.rpg.grammar.Rule;
import foundation.rpg.grammar.Symbol;
import foundation.rpg.processor.ClassToGrammarContext;

import javax.annotation.processing.Filer;
import javax.lang.model.element.ExecutableElement;
import javax.lang.model.type.TypeMirror;
import java.io.IOException;
import java.io.Writer;
import java.util.*;
import java.util.stream.Stream;

import static foundation.rpg.generator.SourceModel.Names.*;
import static java.lang.String.join;
import static java.util.stream.Collectors.*;

public class CodeGenerator {

    private final Filer filer;
    private final ClassToGrammarContext context;

    public CodeGenerator(Filer filer, ClassToGrammarContext context) {
        this.filer = filer;
        this.context = context;
    }

    public void generateSources(LrParserAutomata lrParser) {
        generateState(lrParser);
        write(source("StackState"));
        for(LrItemSet set : lrParser.getSets())
            generateState(lrParser, set);
        Stream.of(lrParser.getGrammar().getTerminals(), lrParser.getGrammar().getIgnored()).flatMap(Collection::stream)
                .forEach(s -> write(source("Token$name$").set(name, s).set(type, typeOf(s))));
    }

    private String chainedType(List<TypeMirror> parameters, int length, int noWildcard) {
        return length > 0 ? "StackState<" + parameters.get(length - 1) + ", " + chainedType(parameters, length - 1, noWildcard - 1) + ">" : noWildcard > 0 ? "State" : "? extends State";
    }

    private String chainedVar(List<TypeMirror> parameters, int length) {
        return chainedType(parameters, length, 1);
    }
    private String chainedType(List<TypeMirror> parameters, int length) {
        return chainedType(parameters, length, 2);
    }

    private void generateState(LrParserAutomata parser) {
        SourceModel code = source("State").set(grammar, parser.getGrammar()).set(automata, parser).set(result, typeOf(parser.getGrammar().getStart()));
        parser.getGrammar().getIgnored().forEach(s -> code.with(Ignored).set(name, s).set(type, typeOf(s)));
        parser.getGrammar().getSymbols().stream().filter(s -> !s.equals(parser.getGrammar().getStart())).forEach(s -> code.with(Symbols).set(name, s).set(type, typeOf(s)));
        write(code);
    }

    private void write(SourceModel source) {
        String string = source.toString();
        try(Writer writer = filer.createSourceFile(source.get("package") + "." + source.get("class")).openWriter()) {
            writer.write(string);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    private SourceModel source(String name) {
        return SourceModel.source(context.getPackageName(), name);
    }

    private String methodName(ExecutableElement method) {
        return method.getEnclosingElement() + "." + method.getSimpleName();
    }

    private TypeMirror typeOf(Symbol symbol) {
        return context.symbolType(symbol);
    }

    private void generateState(LrParserAutomata parser, LrItemSet set) {
        LrItem longest = Collections.max(set.getItems());
        int dot = longest.getDot();
        List<TypeMirror> longestParameters = longest.getRule().getRight().stream().map(context::symbolType).collect(toList());
        String superClass = dot > 0 ? chainedType(longestParameters, dot) : "State";
        SourceModel code = source("State$item$").set(item, set.getName()).set(lrItem, set).set(parent, superClass);
        if (dot > 0)
            code.with(Stack).set(node, longestParameters.get(dot - 1)).set(prev, chainedVar(longestParameters, dot - 1));

        parser.actionsFor(set).forEach((symbol, action) -> action.accept(new LrAction.LrActionVisitor() {
            @Override public void visitGoto(LrItemSet set) {
                code.with(Shift).set(name, symbol).set(type, typeOf(symbol)).set(next, set.getName());
            }

            @Override public void visitReduction(LrItem item) {
                Rule rule = item.getRule();
                SourceModel fragment = code.with(Reduce).set(name, symbol).set(type, typeOf(symbol)).set(result, rule.getLeft());
                String state = visitCall(item, fragment);
                fragment.set(start, state);
            }

            private String visitCall(LrItem item, SourceModel fragment) {
                Rule rule = item.getRule();
                ExecutableElement method = context.methodOf(rule);
                int size = method.getParameters().size();
                String state = "this";
                StringBuilder b = new StringBuilder();
                String[] p = new String[size];
                for(int i = 1; i <= size; i++) {
                    p[size - i] = state + ".getNode()";
                    b.append("\n\t\t").append(chainedVar(longestParameters, dot - i)).append(" stack").append(i).append(" = ").append(state).append(".getPrev();");
                    state = "stack" + i;
                }
                String call = methodName(method) + "(" + join(", ", p) + ")";
                fragment.set(factory, call).set(parameters, b);
                return state;
            }

            @Override public void visitAccept(LrItem item) {
                code.with(Accept).set(result, typeOf(parser.getGrammar().getStart()));
            }
        }));
        write(code);
    }

}
