/*
 * BSD 2-Clause License
 *
 * Copyright (c) 2021, Ondrej Fischer
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.
 *
 *  Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

package fluent.validation;

/**
 * Factory of ready to use most frequent conditions. There are typical conditions for following categories:
 *
 * 1. General check builders for simple building of new conditions using predicate and description.
 * 2. General object conditions - e.g. isNull, notNull, equalTo, etc.
 * 3. Generalized logical operators (N-ary oneOf instead of binary or, N-ary allOf instead of binary and)
 * 4. Collection (Iterable) conditions + quantifiers
 * 5. Relational and range conditions for comparables
 * 6. String matching conditions (contains/startWith/endsWith as well as regexp matching)
 * 7. Basic XML conditions (XPath, attribute matching)
 * 8. Floating point comparison using a tolerance
 * 9. Builders for composition or collection of criteria.
 */

public final class Checks {

	private Checks() {}

	/**
	 * Check, that a double number is close enough to the expected value, given the explicit tolerance.
	 * This is the standard approach to validation of floating point number representations, because due to
	 * the internal representation in the CPU/FPU, numbers coming from e.g. different calculation, but which
	 * would have the same real value, can differ in the CPU/FPU representation.
	 * Simplest example could be: 1/3 vs. (1 - 2/3). Results of these formulas will differ in computer representation,
	 * although in real math they are equal.
	 *
	 * @param expectedValue Expected value.
	 * @param precision Tolerance, which is still accepted as difference from the expected value.
	 * @return Check of the double value.
	 * @see NumericChecks#closeTo(double, double)
	 */
	public static Check<java.lang.Double> closeTo(double expectedValue, double precision) {
		return NumericChecks.closeTo(expectedValue, precision);
	}

	/**
	 * Check, that a float number is close enough to the expected value, given the explicit tolerance.
	 * This is the standard approach to validation of floating point number representations, because due to
	 * the internal representation in the CPU/FPU, numbers coming from e.g. different calculation, but which
	 * would have the same real value, can differ in the CPU/FPU representation.
	 * Simplest example could be: 1/3 vs. (1 - 2/3). Results of these formulas will differ in computer representation,
	 * although in real math they are equal.
	 *
	 * @param expectedValue Expected value.
	 * @param precision Tolerance, which is still accepted as difference from the expected value.
	 * @return Check of the double value.
	 * @see NumericChecks#closeTo(float, float)
	 */
	public static Check<java.lang.Float> closeTo(float expectedValue, float precision) {
		return NumericChecks.closeTo(expectedValue, precision);
	}

	/**
	 * Check, that a BigDecimal number is close enough to the expected value, given the explicit tolerance.
	 * This is the standard approach to validation of floating point number representations, because due to
	 * the internal representation in the CPU/FPU, numbers coming from e.g. different calculation, but which
	 * would have the same real value, can differ in the CPU/FPU representation.
	 * Simplest example could be: 1/3 vs. (1 - 2/3). Results of these formulas will differ in computer representation,
	 * although in real math they are equal.
	 *
	 * @param expectedValue Expected value.
	 * @param precision Tolerance, which is still accepted as difference from the expected value.
	 * @return Check of the double value.
	 * @see NumericChecks#closeTo(java.math.BigDecimal, java.math.BigDecimal)
	 */
	public static Check<java.math.BigDecimal> closeTo(java.math.BigDecimal expectedValue, java.math.BigDecimal precision) {
		return NumericChecks.closeTo(expectedValue, precision);
	}

	/**
	 * Specific overloaded method for "tolerant" equalTo implementation for Double floating point values.
	 * It's implemented using closeTo() with default tolerance, which is by default 0.000001, but can be specified
	 * via system property to different value.
	 *
	 * @param expectedValue Expected value, from which the actual shouldn't differ more, than by the default tolerance.
	 * @return Check of the double equality.
	 * @see #closeTo(double, double)
	 * @see #DEFAULT_TOLERANCE
	 * @see NumericChecks#equalTo(Double)
	 */
	public static Check<java.lang.Double> equalTo(Double expectedValue) {
		return NumericChecks.equalTo(expectedValue);
	}

	/**
	 * Specific overloaded method for "tolerant" equalTo implementation for Float floating point values.
	 * It's implemented using closeTo() with default tolerance, which is by default 0.000001, but can be specified
	 * via system property to different value.
	 *
	 * @param expectedValue Expected value, from which the actual shouldn't differ more, than by the default tolerance.
	 * @return Check of the float equality.
	 * @see #closeTo(float, float)
	 * @see #DEFAULT_TOLERANCE
	 * @see NumericChecks#equalTo(Float)
	 */
	public static Check<java.lang.Float> equalTo(Float expectedValue) {
		return NumericChecks.equalTo(expectedValue);
	}

	/**
	 * Specific overloaded method for "tolerant" equalTo implementation for BigDecimal real number values.
	 * It's implemented using closeTo() with default tolerance, which is by default 0.000001, but can be specified
	 * via system property to different value.
	 *
	 * @param expectedValue Expected value, from which the actual shouldn't differ more, than by the default tolerance.
	 * @return Check of the BigDecimal equality.
	 * @see #closeTo(BigDecimal, BigDecimal)
	 * @see #DEFAULT_TOLERANCE
	 * @see NumericChecks#equalTo(java.math.BigDecimal)
	 */
	public static Check<java.math.BigDecimal> equalTo(java.math.BigDecimal expectedValue) {
		return NumericChecks.equalTo(expectedValue);
	}

	/**
	 * Check of the string containing double number implemented by composition of parsing double from string and
	 * application of check of the Double value.
	 *
	 * @param check Check of the double value.
	 * @return Check applicable on String.
	 * @see NumericChecks#parseDouble(Check<? super java.lang.Double>)
	 */
	public static Check<java.lang.String> parseDouble(Check<? super java.lang.Double> check) {
		return NumericChecks.parseDouble(check);
	}

	/**
	 * Check of the string containing double number implemented by composition of parsing double from string and
	 * expected Double value.
	 *
	 * @param expectedValue Expected double value.
	 * @return Check applicable on String.
	 * @see NumericChecks#parseDouble(Double)
	 */
	public static Check<java.lang.String> parseDouble(Double expectedValue) {
		return NumericChecks.parseDouble(expectedValue);
	}

	/**
	 * Check of the string containing double number implemented by composition of parsing double from string and
	 * expected Double value.
	 *
	 * @param expectedValue Expected double value.
	 * @return Check applicable on String.
	 * @see NumericChecks#parseDouble(double)
	 */
	public static Check<java.lang.String> parseDouble(double expectedValue) {
		return NumericChecks.parseDouble(expectedValue);
	}

	/**
	 * @see NumericChecks#parseFloat(Check<? super java.lang.Float>)
	 */
	public static Check<java.lang.String> parseFloat(Check<? super java.lang.Float> check) {
		return NumericChecks.parseFloat(check);
	}

	/**
	 * @see NumericChecks#parseFloat(Float)
	 */
	public static Check<java.lang.String> parseFloat(Float expectedValue) {
		return NumericChecks.parseFloat(expectedValue);
	}

	/**
	 * @see NumericChecks#parseFloat(float)
	 */
	public static Check<java.lang.String> parseFloat(float expectedValue) {
		return NumericChecks.parseFloat(expectedValue);
	}

	/**
	 * @see NumericChecks#parseInt(Check<? super java.lang.Integer>)
	 */
	public static Check<java.lang.String> parseInt(Check<? super java.lang.Integer> check) {
		return NumericChecks.parseInt(check);
	}

	/**
	 * @see NumericChecks#parseInt(Integer)
	 */
	public static Check<java.lang.String> parseInt(Integer expectedValue) {
		return NumericChecks.parseInt(expectedValue);
	}

	/**
	 * @see NumericChecks#parseInt(int)
	 */
	public static Check<java.lang.String> parseInt(int expectedValue) {
		return NumericChecks.parseInt(expectedValue);
	}

	/**
	 * @see NumericChecks#parseLong(Check<? super java.lang.Long>)
	 */
	public static Check<java.lang.String> parseLong(Check<? super java.lang.Long> check) {
		return NumericChecks.parseLong(check);
	}

	/**
	 * @see NumericChecks#parseLong(Long)
	 */
	public static Check<java.lang.String> parseLong(Long expectedValue) {
		return NumericChecks.parseLong(expectedValue);
	}

	/**
	 * @see NumericChecks#parseLong(long)
	 */
	public static Check<java.lang.String> parseLong(long expectedValue) {
		return NumericChecks.parseLong(expectedValue);
	}

	/**
	 * @see NumericChecks#parseShort(Check<? super java.lang.Short>)
	 */
	public static Check<java.lang.String> parseShort(Check<? super java.lang.Short> check) {
		return NumericChecks.parseShort(check);
	}

	/**
	 * @see NumericChecks#parseShort(Short)
	 */
	public static Check<java.lang.String> parseShort(Short expectedValue) {
		return NumericChecks.parseShort(expectedValue);
	}

	/**
	 * @see NumericChecks#parseShort(short)
	 */
	public static Check<java.lang.String> parseShort(short expectedValue) {
		return NumericChecks.parseShort(expectedValue);
	}

	/**
	 * @see NumericChecks#parseBigDecimal(Check<? super java.math.BigDecimal>)
	 */
	public static Check<java.lang.String> parseBigDecimal(Check<? super java.math.BigDecimal> check) {
		return NumericChecks.parseBigDecimal(check);
	}

	/**
	 * @see NumericChecks#parseBigInt(Check<? super java.math.BigInteger>)
	 */
	public static Check<java.lang.String> parseBigInt(Check<? super java.math.BigInteger> check) {
		return NumericChecks.parseBigInt(check);
	}

	/**
	 * @see NumericChecks#parseByte(Check<? super java.lang.Byte>)
	 */
	public static Check<java.lang.String> parseByte(Check<? super java.lang.Byte> check) {
		return NumericChecks.parseByte(check);
	}

	/**
	 * @see NumericChecks#parseByte(Byte)
	 */
	public static Check<java.lang.String> parseByte(Byte expectedValue) {
		return NumericChecks.parseByte(expectedValue);
	}

	/**
	 * @see NumericChecks#parseByte(byte)
	 */
	public static Check<java.lang.String> parseByte(byte expectedValue) {
		return NumericChecks.parseByte(expectedValue);
	}

	/**
	 * @see NumericChecks#parseBoolean(Check<? super java.lang.Boolean>)
	 */
	public static Check<java.lang.String> parseBoolean(Check<? super java.lang.Boolean> check) {
		return NumericChecks.parseBoolean(check);
	}

	/**
	 * @see NumericChecks#parseBoolean(Boolean)
	 */
	public static Check<java.lang.String> parseBoolean(Boolean expectedValue) {
		return NumericChecks.parseBoolean(expectedValue);
	}

	/**
	 * Define a transparent check using provided predicate and string expectation description.
	 *
	 * @param predicate Predicate used to test the supplied data.
	 * @param expectationDescription Function, that provides description of the expectation.
	 * @param <D> Type of the data to be tested by the created expectation.
	 * @return New expectation.
	 * @see BasicChecks#nullableCheck(Predicate<D>, String)
	 */
	public static <D extends java.lang.Object> Check<D> nullableCheck(Predicate<D> predicate, String expectationDescription) {
		return BasicChecks.nullableCheck(predicate, expectationDescription);
	}

	/**
	 * Require, that tested data meet provided requirement before application of a check.
	 * If the requirement is not met, composed check returns false, and the "actual" check is not evaluated.
	 * If the requirement is met, then "actual" check is evaluated, and the result is returned.
	 *
	 * Error description:
	 * If requirement is not met, then error contains description of it's mismatch.
	 * If requirement is met, then error contains only mismatch of the "actual" check, while requirement stays silent.
	 *
	 * The behavior is like Java's and operator, which doesn't evaluate second operand if first one is false.
	 * The and() operator check as opposed to this one, evaluates always both operands to get mismatch details from
	 * both of them. Keep that in mind and properly decide, in which situation to use which operator.
	 *
	 * @param requirement Check, that must return true in order to evaluate the "actual" one.
	 * @param check "Actual" check to be evaluated on tested data, if requirement is met.
	 * @param <D> Type of the data to be tested by the created check.
	 * @return Composed check.
	 * @see BasicChecks#require(Check<? super D>, Check<? super D>)
	 */
	public static <D extends java.lang.Object> Check<D> require(Check<? super D> requirement, Check<? super D> check) {
		return BasicChecks.require(requirement, check);
	}

	/**
	 * Require, that tested value is not null, before application of the provided check on it.
	 * If the tested value is null, then it returns false without evaluating the check.
	 * If the tested value is not null, then the provided check is applied, and result is returned.
	 *
	 * @param check Check to apply on the value, if it's not null.
	 * @param <D> Type of the data to be tested by the created check.
	 * @return Composed check.
	 * @see BasicChecks#requireNotNull(Check<D>)
	 */
	public static <D extends java.lang.Object> Check<D> requireNotNull(Check<D> check) {
		return BasicChecks.requireNotNull(check);
	}

	/**
	 * Build a custom check using it's description and pure Java predicate. As the Predicate doesn't
	 * provide any transparency, it's enriched with the description.
	 *
	 * @param predicate Implementation of the check logic.
	 * @param expectationDescription Description of the expectation represented by this check.
	 * @param <D> Type of the data to be checked.
	 * @return Custom check.
	 * @see BasicChecks#check(Predicate<D>, String)
	 */
	public static <D extends java.lang.Object> Check<D> check(Predicate<D> predicate, String expectationDescription) {
		return BasicChecks.check(predicate, expectationDescription);
	}

	/**
	 * Create a check, that checks if the tested value is exactly the same instance as expected.
	 * This check doesn't use `equals()` method, but reference comparison operator ==.
	 *
	 * @param expectedInstance Expected instance.
	 * @param <D> Type of the tested data.
	 * @return New check.
	 * @see BasicChecks#sameInstance(D)
	 */
	public static <D extends java.lang.Object> Check<D> sameInstance(D expectedInstance) {
		return BasicChecks.sameInstance(expectedInstance);
	}

	/**
	 * Create a check, that checks if the tested value is equal to expected value.
	 * It will return true as long as the equals() method does.
	 *
	 * @param expectedValue Expected value.
	 * @param <D> Type of the tested data.
	 * @return New check.
	 * @see BasicChecks#equalTo(D)
	 */
	public static <D extends java.lang.Object> Check<D> equalTo(D expectedValue) {
		return BasicChecks.equalTo(expectedValue);
	}

	/**
	 * Alias for `equalTo`, which may be more readable in some constructions.
	 *
	 * @param expectedValue Expected value.
	 * @param <D> Type of the tested data.
	 * @return New check.
	 * @see #equalTo(Object)
	 * @see BasicChecks#is(D)
	 */
	public static <D extends java.lang.Object> Check<D> is(D expectedValue) {
		return BasicChecks.is(expectedValue);
	}

	/**
	 * Check, if tested data is null.
	 * @return New check.
	 * @see BasicChecks#isNull()
	 */
	public static Check<java.lang.Object> isNull() {
		return BasicChecks.isNull();
	}

	/**
	 * Check, if tested data is not null.
	 * @return New check.
	 * @see BasicChecks#isNotNull()
	 */
	public static Check<java.lang.Object> isNotNull() {
		return BasicChecks.isNotNull();
	}

	/**
	 * Check, that doesn't care of the tested value, and always returns true.
	 * @return New check.
	 * @see BasicChecks#anything()
	 */
	public static Check<java.lang.Object> anything() {
		return BasicChecks.anything();
	}

	/**
	 * Check, that tested data is instance of provided class (either the class itself or any subclass).
	 *
	 * @param expectedClass Expected class.
	 * @return New check.
	 * @see BasicChecks#instanceOf(Class<?>)
	 */
	public static Check<java.lang.Object> instanceOf(Class<?> expectedClass) {
		return BasicChecks.instanceOf(expectedClass);
	}

	/**
	 * Alias of `instanceOf` which may be more readable in some context.
	 *
	 * @param expectedClass Expected class.
	 * @return New check.
	 * @see BasicChecks#isA(Class<?>)
	 */
	public static Check<java.lang.Object> isA(Class<?> expectedClass) {
		return BasicChecks.isA(expectedClass);
	}

	/**
	 * Alias of `instanceOf` which may be more readable in some context.
	 *
	 * @param expectedClass Expected class.
	 * @return New check.
	 * @see BasicChecks#isAn(Class<?>)
	 */
	public static Check<java.lang.Object> isAn(Class<?> expectedClass) {
		return BasicChecks.isAn(expectedClass);
	}

	/**
	 * Alias of `instanceOf` which may be more readable in some context.
	 *
	 * @param expectedClass Expected class.
	 * @return New check.
	 * @see BasicChecks#a(Class<?>)
	 */
	public static Check<java.lang.Object> a(Class<?> expectedClass) {
		return BasicChecks.a(expectedClass);
	}

	/**
	 * Alias of `instanceOf` which may be more readable in some context.
	 *
	 * @param expectedClass Expected class.
	 * @return New check.
	 * @see BasicChecks#an(Class<?>)
	 */
	public static Check<java.lang.Object> an(Class<?> expectedClass) {
		return BasicChecks.an(expectedClass);
	}

	/**
	 * Check, that tested data is exactly of the provided type, but not any subcalss of it.
	 * @param expectedClass Expected class.
	 * @return New check.
	 * @see BasicChecks#sameClass(Class<?>)
	 */
	public static Check<java.lang.Object> sameClass(Class<?> expectedClass) {
		return BasicChecks.sameClass(expectedClass);
	}

	/**
	 * Check, that provided positive check doesn't match checked data.
	 *
	 * @param positiveCheck Positive check to evaluate, what shouldn't match.
	 * @param <D> Type of the data to be tested by the created check.
	 * @return Negated expectation.
	 * @see BasicChecks#not(Check<D>)
	 */
	public static <D extends java.lang.Object> Check<D> not(Check<D> positiveCheck) {
		return BasicChecks.not(positiveCheck);
	}

	/**
	 * Check, that provided positive value isn't equal to checked data.
	 *
	 * @param positiveValue Positive value, which shouldn't be equal to checked data.
	 * @param <D> Type of the data to be tested by the created check.
	 * @return Negated expectation.
	 * @see BasicChecks#not(D)
	 */
	public static <D extends java.lang.Object> Check<D> not(D positiveValue) {
		return BasicChecks.not(positiveValue);
	}

	/**
	 * Create matcher, that matches, if tested object meets one of provided alternatives (or).
	 *
	 * @param alternatives Expected alternatives.
	 * @param <D> Type of the data to test using this expectation.
	 * @return Expectation with alternatives.
	 * @see BasicChecks#oneOf(java.util.Collection<D>)
	 */
	public static <D extends java.lang.Object> Check<D> oneOf(java.util.Collection<D> alternatives) {
		return BasicChecks.oneOf(alternatives);
	}

	/**
	 * Create matcher, that matches, if tested object meets one of provided alternatives (or).
	 *
	 * @param alternatives Expected alternatives.
	 * @param <D> Type of the data to test using this expectation.
	 * @return Expectation with alternatives.
	 * @see BasicChecks#oneOf(D[])
	 */
	@SafeVarargs
	public static <D extends java.lang.Object> Check<D> oneOf(D... alternatives) {
		return BasicChecks.oneOf(alternatives);
	}

	/**
	 * General OR operator of multiple checks.
	 * The created check returns true if any of the provided operands (checks) returns true on the tested data,
	 * and only if none of the operands returns true, then the resulting check returns false.
	 *
	 * Corner case:
	 * If no operands are provided, then it returns false (as none of them evaluated to true).
	 *
	 * @param operands Individual checks to evaluate and apply logical or on their results.
	 * @param <D> Type of the tested data.
	 * @return New check with the described logic.
	 * @see BasicChecks#anyOf(Iterable<fluent.validation.Check<? super D>>)
	 */
	public static <D extends java.lang.Object> Check<D> anyOf(Iterable<fluent.validation.Check<? super D>> operands) {
		return BasicChecks.anyOf(operands);
	}

	/**
	 * General OR operator of multiple checks.
	 * The created check returns true if any of the provided operands (checks) returns true on the tested data,
	 * and only if none of the operands returns true, then the resulting check returns false.
	 *
	 * Corner case:
	 * If no operands are provided, then it returns false (as none of them evaluated to true).
	 *
	 * @param operands Individual checks to evaluate and apply logical or on their results.
	 * @param <D> Type of the tested data.
	 * @return New check with the described logic.
	 * @see BasicChecks#anyOf(fluent.validation.Check<? super D>[])
	 */
	@SafeVarargs
	public static <D extends java.lang.Object> Check<D> anyOf(Check<? super D>... operands) {
		return BasicChecks.anyOf(operands);
	}

	/**
	 * General AND operator of multiple checks.
	 * The created check returns true if all the provided operands (checks) return true on the tested data,
	 * and if any of the operands returns false, then the resulting check returns false.
	 *
	 * Corner case:
	 * If no operands are provided, then it returns true (as none of them evaluated to false).
	 *
	 * @param operands Individual checks to evaluate and apply logical and on their results.
	 * @param <D> Type of the tested data.
	 * @return New check with the described logic.
	 * @see BasicChecks#allOf(Iterable<fluent.validation.Check<? super D>>)
	 */
	public static <D extends java.lang.Object> Check<D> allOf(Iterable<fluent.validation.Check<? super D>> operands) {
		return BasicChecks.allOf(operands);
	}

	/**
	 * General AND operator of multiple checks.
	 * The created check returns true if all the provided operands (checks) return true on the tested data,
	 * and if any of the operands returns false, then the resulting check returns false.
	 *
	 * Corner case:
	 * If no operands are provided, then it returns true (as none of them evaluated to false).
	 *
	 * @param operands Individual checks to evaluate and apply logical and on their results.
	 * @param <D> Type of the tested data.
	 * @return New check with the described logic.
	 * @see BasicChecks#allOf(fluent.validation.Check<? super D>[])
	 */
	@SafeVarargs
	public static <D extends java.lang.Object> Check<D> allOf(Check<? super D>... operands) {
		return BasicChecks.allOf(operands);
	}

	/**
	 * Lowest level transformation.
	 * Create a new check using transformation of the original object, e.g. by accessing it's field, and partial check
	 * applied on the result of the transformation.
	 *
	 * @param transformation Function transforming the original object.
	 * @param check Check to apply on the result of the transformation.
	 * @param <D> Type of the original tested object.
	 * @param <V> Type of the transformation result.
	 * @return Composed check applicable on the original (whole) object.
	 * @see BasicChecks#transform(Transformation<? super D,V>, Check<? super V>)
	 */
	public static <D extends java.lang.Object, V extends java.lang.Object> Check<D> transform(Transformation<? super D,V> transformation, Check<? super V> check) {
		return BasicChecks.transform(transformation, check);
	}

	/**
	 * Convenient composition using transformation of the original object, and check on the result.
	 *
	 * @param name Name (description) of the transformation.
	 * @param transformation Function transforming the original object.
	 * @param check Check to apply on the result of the transformation.
	 * @param <D> Type of the original tested object.
	 * @param <V> Type of the transformation result.
	 * @return Composed check applicable on the original (whole) object.
	 * @see BasicChecks#compose(String, Transformation<? super D,V>, Check<? super V>)
	 */
	public static <D extends java.lang.Object, V extends java.lang.Object> Check<D> compose(String name, Transformation<? super D,V> transformation, Check<? super V> check) {
		return BasicChecks.compose(name, transformation, check);
	}

	/**
	 * Simplified composition using transformation of the original object, and check on the result.
	 *
	 * @param transformation Function transforming the original object.
	 * @param check Check to apply on the result of the transformation.
	 * @param <D> Type of the original tested object.
	 * @param <V> Type of the transformation result.
	 * @return Composed check applicable on the original (whole) object.
	 * @see BasicChecks#compose(Transformation<? super D,V>, Check<? super V>)
	 */
	public static <D extends java.lang.Object, V extends java.lang.Object> Check<D> compose(Transformation<? super D,V> transformation, Check<? super V> check) {
		return BasicChecks.compose(transformation, check);
	}

	/**
	 * @see BasicChecks#has(String, Transformation<? super D,V>)
	 */
	public static <D extends java.lang.Object, V extends java.lang.Object> TransformationBuilder<V,fluent.validation.CheckBuilder<D>> has(String name, Transformation<? super D,V> transformation) {
		return BasicChecks.has(name, transformation);
	}

	/**
	 * @see BasicChecks#has(Transformation<? super D,V>)
	 */
	public static <D extends java.lang.Object, V extends java.lang.Object> TransformationBuilder<V,fluent.validation.CheckBuilder<D>> has(Transformation<? super D,V> transformation) {
		return BasicChecks.has(transformation);
	}

	/**
	 * @see BasicChecks#nullableHas(String, Transformation<? super D,V>)
	 */
	public static <D extends java.lang.Object, V extends java.lang.Object> TransformationBuilder<V,fluent.validation.CheckBuilder<D>> nullableHas(String name, Transformation<? super D,V> transformation) {
		return BasicChecks.nullableHas(name, transformation);
	}

	/**
	 * @see BasicChecks#nullableHas(Transformation<? super D,V>)
	 */
	public static <D extends java.lang.Object, V extends java.lang.Object> TransformationBuilder<V,fluent.validation.CheckBuilder<D>> nullableHas(Transformation<? super D,V> transformation) {
		return BasicChecks.nullableHas(transformation);
	}

	/**
	 * Fluent builder of a check, that will assure, that provided object is instance of required class, and if yes,
	 * it will cast it to that class, and apply provided check on the required type.
	 *
	 * @param type Required Java class / type
	 * @param <V> Type to check, and cast to.
	 * @return Builder of a object check, that accepts check on the required type.
	 * @see BasicChecks#as(Class<V>)
	 */
	public static <V extends java.lang.Object> TransformationBuilder<V,fluent.validation.Check<V>> as(Class<V> type) {
		return BasicChecks.as(type);
	}

	/**
	 * Check of exception (throwable) message.
	 *
	 * @param check String check of the message content.
	 * @return Check on Throwable.
	 * @see BasicChecks#message(Check<? super java.lang.String>)
	 */
	public static Check<java.lang.Throwable> message(Check<? super java.lang.String> check) {
		return BasicChecks.message(check);
	}

	/**
	 * Check of a code, which should throw an exception (throwable).
	 * The created check will actually run the code passed to test, and return true if the code throws exception, which
	 * matches specified criteria.
	 *
	 * @param check Check on the throwable.
	 * @return ThrowingCheck applicable on provided code, allowing to build further criteria.
	 * @see BasicChecks#throwing(Check<? super java.lang.Throwable>)
	 */
	public static ThrowingCheck throwing(Check<? super java.lang.Throwable> check) {
		return BasicChecks.throwing(check);
	}

	/**
	 * Check of a code, which should throw an exception (throwable) of expected type.
	 * The created check will actually run the code passed to test, and return true if the code throws exception of the
	 * required type.
	 *
	 * @param expectedType Expected type of the thrown exception.
	 * @return ThrowingCheck applicable on provided code, allowing to build further criteria.
	 * @see BasicChecks#throwing(Class<? extends java.lang.Throwable>)
	 */
	public static ThrowingCheck throwing(Class<? extends java.lang.Throwable> expectedType) {
		return BasicChecks.throwing(expectedType);
	}

	/**
	 * @see BasicChecks#createBuilder()
	 */
	public static <D extends java.lang.Object> Check<D> createBuilder() {
		return BasicChecks.createBuilder();
	}

	/**
	 * @see BasicChecks#createBuilderWith(Check<D>)
	 */
	public static <D extends java.lang.Object> Check<D> createBuilderWith(Check<D> check) {
		return BasicChecks.createBuilderWith(check);
	}

	/**
	 * @see BasicChecks#which(Check<D>)
	 */
	public static <D extends java.lang.Object> Check<D> which(Check<D> check) {
		return BasicChecks.which(check);
	}

	/**
	 * @see BasicChecks#dsl()
	 */
	public static <D extends java.lang.Object> CheckBuilder<D> dsl() {
		return BasicChecks.dsl();
	}

	/**
	 * @see BasicChecks#value()
	 */
	public static <D extends java.lang.Object> Value<D> value() {
		return BasicChecks.value();
	}

	/**
	 * @see BasicChecks#softCheck(Check<D>)
	 */
	public static <D extends java.lang.Object> Check<D> softCheck(Check<D> check) {
		return BasicChecks.softCheck(check);
	}

	/**
	 * @see BasicChecks#customResultFactory(Check<D>, fluent.validation.result.ResultFactory)
	 */
	public static <D extends java.lang.Object> Check<D> customResultFactory(Check<D> check, fluent.validation.result.ResultFactory customResultFactory) {
		return BasicChecks.customResultFactory(check, customResultFactory);
	}

	/**
	 * @see DateTimeChecks#date(Check<? super java.time.LocalDate>)
	 */
	public static Check<java.time.LocalDateTime> date(Check<? super java.time.LocalDate> dateCheck) {
		return DateTimeChecks.date(dateCheck);
	}

	/**
	 * @see DateTimeChecks#time(Check<? super java.time.LocalTime>)
	 */
	public static Check<java.time.LocalDateTime> time(Check<? super java.time.LocalTime> dateCheck) {
		return DateTimeChecks.time(dateCheck);
	}

	/**
	 * @see DateTimeChecks#zoned(Check<? super java.time.ZonedDateTime>, java.time.ZoneId)
	 */
	public static Check<java.time.LocalDateTime> zoned(Check<? super java.time.ZonedDateTime> zonedDateTimeCheck, java.time.ZoneId zoneId) {
		return DateTimeChecks.zoned(zonedDateTimeCheck, zoneId);
	}

	/**
	 * @see DateTimeChecks#zoned(Check<? super java.time.ZonedDateTime>)
	 */
	public static Check<java.time.LocalDateTime> zoned(Check<? super java.time.ZonedDateTime> zonedDateTimeCheck) {
		return DateTimeChecks.zoned(zonedDateTimeCheck);
	}

	/**
	 * @see DateTimeChecks#localDateTimeWith()
	 */
	public static fluent.validation.dsl.LocalDateTimeCheck localDateTimeWith() {
		return DateTimeChecks.localDateTimeWith();
	}

	/**
	 * @see DateTimeChecks#localDateWith()
	 */
	public static fluent.validation.dsl.LocalDateCheck localDateWith() {
		return DateTimeChecks.localDateWith();
	}

	/**
	 * @see DateTimeChecks#localTimeWith()
	 */
	public static fluent.validation.dsl.LocalTimeCheck localTimeWith() {
		return DateTimeChecks.localTimeWith();
	}

	/**
	 * @see DateTimeChecks#zonedDateTimeWith()
	 */
	public static fluent.validation.dsl.ZonedDateTimeCheck zonedDateTimeWith() {
		return DateTimeChecks.zonedDateTimeWith();
	}

	/**
	 * @see DateTimeChecks#parseLocalDateTime(Check<? super java.time.LocalDateTime>)
	 */
	public static Check<java.lang.String> parseLocalDateTime(Check<? super java.time.LocalDateTime> check) {
		return DateTimeChecks.parseLocalDateTime(check);
	}

	/**
	 * @see DateTimeChecks#parseLocalDateTime(String, Check<? super java.time.LocalDateTime>)
	 */
	public static Check<java.lang.String> parseLocalDateTime(String format, Check<? super java.time.LocalDateTime> check) {
		return DateTimeChecks.parseLocalDateTime(format, check);
	}

	/**
	 * @see DateTimeChecks#parseLocalDateTime(java.time.format.DateTimeFormatter, Check<? super java.time.LocalDateTime>)
	 */
	public static Check<java.lang.String> parseLocalDateTime(java.time.format.DateTimeFormatter format, Check<? super java.time.LocalDateTime> check) {
		return DateTimeChecks.parseLocalDateTime(format, check);
	}

	/**
	 * @see DateTimeChecks#parseZonedDateTime(Check<? super java.time.ZonedDateTime>)
	 */
	public static Check<java.lang.String> parseZonedDateTime(Check<? super java.time.ZonedDateTime> check) {
		return DateTimeChecks.parseZonedDateTime(check);
	}

	/**
	 * @see DateTimeChecks#parseZonedDateTime(String, Check<? super java.time.ZonedDateTime>)
	 */
	public static Check<java.lang.String> parseZonedDateTime(String format, Check<? super java.time.ZonedDateTime> check) {
		return DateTimeChecks.parseZonedDateTime(format, check);
	}

	/**
	 * @see DateTimeChecks#parseZonedDateTime(java.time.format.DateTimeFormatter, Check<? super java.time.ZonedDateTime>)
	 */
	public static Check<java.lang.String> parseZonedDateTime(java.time.format.DateTimeFormatter format, Check<? super java.time.ZonedDateTime> check) {
		return DateTimeChecks.parseZonedDateTime(format, check);
	}

	/**
	 * @see CollectionChecks#exists(String, Check<? super D>)
	 */
	public static <D extends java.lang.Object> Check<java.lang.Iterable<D>> exists(String elementName, Check<? super D> check) {
		return CollectionChecks.exists(elementName, check);
	}

	/**
	 * @see CollectionChecks#every(String, Check<? super D>)
	 */
	public static <D extends java.lang.Object> Check<java.lang.Iterable<D>> every(String elementName, Check<? super D> check) {
		return CollectionChecks.every(elementName, check);
	}

	/**
	 * @see CollectionChecks#repeatMax(Check<D>, int)
	 */
	public static <D extends java.lang.Object> Check<D> repeatMax(Check<D> attemptCheck, int max) {
		return CollectionChecks.repeatMax(attemptCheck, max);
	}

	/**
	 * @see CollectionChecks#repeatMax(Check<D>, int, java.time.Duration)
	 */
	public static <D extends java.lang.Object> Check<D> repeatMax(Check<D> itemCheck, int max, java.time.Duration delay) {
		return CollectionChecks.repeatMax(itemCheck, max, delay);
	}

	/**
	 * @see CollectionChecks#startsWith(String, Items<T>)
	 */
	public static <T extends java.lang.Object> Check<java.util.Iterator<T>> startsWith(String elementName, Items<T> prefix) {
		return CollectionChecks.startsWith(elementName, prefix);
	}

	/**
	 * @see CollectionChecks#contains(String, Items<T>)
	 */
	public static <T extends java.lang.Object> Check<java.util.Iterator<T>> contains(String elementName, Items<T> elementChecks) {
		return CollectionChecks.contains(elementName, elementChecks);
	}

	/**
	 * @see CollectionChecks#equalTo(String, Items<T>)
	 */
	public static <T extends java.lang.Object> Check<java.util.Iterator<T>> equalTo(String elementName, Items<T> elementChecks) {
		return CollectionChecks.equalTo(elementName, elementChecks);
	}

	/**
	 * @see CollectionChecks#startsWith(Items<T>)
	 */
	public static <T extends java.lang.Object> Check<java.util.Iterator<T>> startsWith(Items<T> prefix) {
		return CollectionChecks.startsWith(prefix);
	}

	/**
	 * @see CollectionChecks#contains(Items<T>)
	 */
	public static <T extends java.lang.Object> Check<java.util.Iterator<T>> contains(Items<T> elementChecks) {
		return CollectionChecks.contains(elementChecks);
	}

	/**
	 * @see CollectionChecks#equalTo(Items<T>)
	 */
	public static <T extends java.lang.Object> Check<java.util.Iterator<T>> equalTo(Items<T> elementChecks) {
		return CollectionChecks.equalTo(elementChecks);
	}

	/**
	 * @see CollectionChecks#startsInAnyOrderWith(String, Items<T>)
	 */
	public static <T extends java.lang.Object> Check<java.util.Iterator<T>> startsInAnyOrderWith(String elementName, Items<T> prefixElementChecks) {
		return CollectionChecks.startsInAnyOrderWith(elementName, prefixElementChecks);
	}

	/**
	 * @see CollectionChecks#containsInAnyOrder(String, Items<T>)
	 */
	public static <T extends java.lang.Object> Check<java.util.Iterator<T>> containsInAnyOrder(String elementName, Items<T> elementChecks) {
		return CollectionChecks.containsInAnyOrder(elementName, elementChecks);
	}

	/**
	 * @see CollectionChecks#containsInAnyOrderOnly(String, Items<T>)
	 */
	public static <T extends java.lang.Object> Check<java.util.Iterator<T>> containsInAnyOrderOnly(String elementName, Items<T> elementChecks) {
		return CollectionChecks.containsInAnyOrderOnly(elementName, elementChecks);
	}

	/**
	 * @see CollectionChecks#equalInAnyOrderTo(String, Items<T>)
	 */
	public static <T extends java.lang.Object> Check<java.util.Iterator<T>> equalInAnyOrderTo(String elementName, Items<T> elementChecks) {
		return CollectionChecks.equalInAnyOrderTo(elementName, elementChecks);
	}

	/**
	 * @see CollectionChecks#startsInAnyOrderWith(Items<T>)
	 */
	public static <T extends java.lang.Object> Check<java.util.Iterator<T>> startsInAnyOrderWith(Items<T> prefixElementChecks) {
		return CollectionChecks.startsInAnyOrderWith(prefixElementChecks);
	}

	/**
	 * @see CollectionChecks#containsInAnyOrder(Items<T>)
	 */
	public static <T extends java.lang.Object> Check<java.util.Iterator<T>> containsInAnyOrder(Items<T> elementChecks) {
		return CollectionChecks.containsInAnyOrder(elementChecks);
	}

	/**
	 * @see CollectionChecks#containsInAnyOrderOnly(Items<T>)
	 */
	public static <T extends java.lang.Object> Check<java.util.Iterator<T>> containsInAnyOrderOnly(Items<T> elementChecks) {
		return CollectionChecks.containsInAnyOrderOnly(elementChecks);
	}

	/**
	 * @see CollectionChecks#equalInAnyOrderTo(Items<T>)
	 */
	public static <T extends java.lang.Object> Check<java.util.Iterator<T>> equalInAnyOrderTo(Items<T> elementChecks) {
		return CollectionChecks.equalInAnyOrderTo(elementChecks);
	}

	/**
	 * @see CollectionChecks#collection(Check<java.util.Iterator<T>>)
	 */
	public static <T extends java.lang.Object> Check<java.lang.Iterable<T>> collection(Check<java.util.Iterator<T>> check) {
		return CollectionChecks.collection(check);
	}

	/**
	 * @see CollectionChecks#array(Check<java.util.Iterator<T>>)
	 */
	public static <T extends java.lang.Object> Check<T[]> array(Check<java.util.Iterator<T>> check) {
		return CollectionChecks.array(check);
	}

	/**
	 * @see CollectionChecks#queue(Check<java.util.Iterator<T>>)
	 */
	public static <T extends java.lang.Object> Check<java.util.Queue<T>> queue(Check<java.util.Iterator<T>> check) {
		return CollectionChecks.queue(check);
	}

	/**
	 * @see CollectionChecks#blockingQueue(Check<java.util.Iterator<T>>, java.time.Duration)
	 */
	public static <T extends java.lang.Object> Check<java.util.concurrent.BlockingQueue<T>> blockingQueue(Check<java.util.Iterator<T>> check, java.time.Duration timeout) {
		return CollectionChecks.blockingQueue(check, timeout);
	}

	/**
	 * Create matcher of empty collection.
	 * It returns true, if tested collection meets null or has no elements.
	 *
	 * @param <D> Type of the items in the collection to be tested.
	 * @return Empty collection expectation.
	 * @see CollectionChecks#emptyCollection()
	 */
	public static <D extends java.lang.Object> Check<java.util.Collection<D>> emptyCollection() {
		return CollectionChecks.emptyCollection();
	}

	/**
	 * Create matcher, that matches if tested collection meets subset of provided superset.
	 *
	 * @param superSet Superset of expected items.
	 * @param <D> Type of the items in the collection to be tested.
	 * @return Subset expectation.
	 * @see CollectionChecks#subsetOf(java.util.Collection<D>)
	 */
	public static <D extends java.lang.Object> Check<java.util.Collection<D>> subsetOf(java.util.Collection<D> superSet) {
		return CollectionChecks.subsetOf(superSet);
	}

	/**
	 * Create matcher, that matches if tested collection meets subset of provided superset.
	 *
	 * @param superSet Superset of expected items.
	 * @param <D> Type of the items in the collection to be tested.
	 * @return Subset expectation.
	 * @see CollectionChecks#subsetOf(D[])
	 */
	@SafeVarargs
	public static <D extends java.lang.Object> Check<java.util.Collection<D>> subsetOf(D... superSet) {
		return CollectionChecks.subsetOf(superSet);
	}

	/**
	 * Create matcher of collection size.
	 * If you want to test for emptiness using size 0, note different behavior from empty() for null. This expectation
	 * will fail for it, but empty() will pass.
	 *
	 * @param size Expected size of the collection.
	 * @param <D> Type of the items in the collection to be tested.
	 * @return Collection size expectation.
	 * @see CollectionChecks#hasSize(int)
	 */
	public static <D extends java.lang.Object> Check<java.util.Collection<D>> hasSize(int size) {
		return CollectionChecks.hasSize(size);
	}

	/**
	 * @see CollectionChecks#containsAll(java.util.Collection<D>)
	 */
	public static <D extends java.lang.Object> Check<java.util.Collection<D>> containsAll(java.util.Collection<D> items) {
		return CollectionChecks.containsAll(items);
	}

	/**
	 * @see CollectionChecks#containsAll(D[])
	 */
	@SafeVarargs
	public static <D extends java.lang.Object> Check<java.util.Collection<D>> containsAll(D... items) {
		return CollectionChecks.containsAll(items);
	}

	/**
	 * @see CollectionChecks#mapHas(K, Check<? super V>)
	 */
	public static <K extends java.lang.Object, V extends java.lang.Object> Check<java.util.Map<K,V>> mapHas(K key, Check<? super V> check) {
		return CollectionChecks.mapHas(key, check);
	}

	/**
	 * Create matcher of empty array.
	 * It returns true, if tested collection meets null or has no elements.
	 *
	 * @param <D> Type of the items in the tested array.
	 * @return Empty array expectation.
	 * @see CollectionChecks#emptyArray()
	 */
	public static <D extends java.lang.Object> Check<D[]> emptyArray() {
		return CollectionChecks.emptyArray();
	}

	/**
	 * Check, that provided data is either null or empty array.
	 *
	 * @param <D> Type of the items in the tested array.
	 * @return Empty array expectation.
	 * @see CollectionChecks#emptyArrayOrNull()
	 */
	public static <D extends java.lang.Object> Check<D[]> emptyArrayOrNull() {
		return CollectionChecks.emptyArrayOrNull();
	}

	/**
	 * @see ComparisonChecks#lessThan(D, java.util.Comparator<D>)
	 */
	public static <D extends java.lang.Object> Check<D> lessThan(D operand, java.util.Comparator<D> comparator) {
		return ComparisonChecks.lessThan(operand, comparator);
	}

	/**
	 * @see ComparisonChecks#moreThan(D, java.util.Comparator<D>)
	 */
	public static <D extends java.lang.Object> Check<D> moreThan(D operand, java.util.Comparator<D> comparator) {
		return ComparisonChecks.moreThan(operand, comparator);
	}

	/**
	 * @see ComparisonChecks#equalOrLessThan(D, java.util.Comparator<D>)
	 */
	public static <D extends java.lang.Object> Check<D> equalOrLessThan(D operand, java.util.Comparator<D> comparator) {
		return ComparisonChecks.equalOrLessThan(operand, comparator);
	}

	/**
	 * @see ComparisonChecks#equalOrMoreThan(D, java.util.Comparator<D>)
	 */
	public static <D extends java.lang.Object> Check<D> equalOrMoreThan(D operand, java.util.Comparator<D> comparator) {
		return ComparisonChecks.equalOrMoreThan(operand, comparator);
	}

	/**
	 * @see ComparisonChecks#lessThan(D)
	 */
	public static <D extends java.lang.Comparable<D>> Check<D> lessThan(D operand) {
		return ComparisonChecks.lessThan(operand);
	}

	/**
	 * @see ComparisonChecks#moreThan(D)
	 */
	public static <D extends java.lang.Comparable<D>> Check<D> moreThan(D operand) {
		return ComparisonChecks.moreThan(operand);
	}

	/**
	 * @see ComparisonChecks#equalOrLessThan(D)
	 */
	public static <D extends java.lang.Comparable<D>> Check<D> equalOrLessThan(D operand) {
		return ComparisonChecks.equalOrLessThan(operand);
	}

	/**
	 * @see ComparisonChecks#equalOrMoreThan(D)
	 */
	public static <D extends java.lang.Comparable<D>> Check<D> equalOrMoreThan(D operand) {
		return ComparisonChecks.equalOrMoreThan(operand);
	}

	/**
	 * @see ComparisonChecks#between(D, D)
	 */
	public static <D extends java.lang.Comparable<D>> Check<D> between(D left, D right) {
		return ComparisonChecks.between(left, right);
	}

	/**
	 * @see ComparisonChecks#between(D, D, java.util.Comparator<D>)
	 */
	public static <D extends java.lang.Object> Check<D> between(D left, D right, java.util.Comparator<D> comparator) {
		return ComparisonChecks.between(left, right, comparator);
	}

	/**
	 * @see ComparisonChecks#betweenInclusive(D, D, java.util.Comparator<D>)
	 */
	public static <D extends java.lang.Object> Check<D> betweenInclusive(D left, D right, java.util.Comparator<D> comparator) {
		return ComparisonChecks.betweenInclusive(left, right, comparator);
	}

	/**
	 * @see ComparisonChecks#betweenInclusive(D, D)
	 */
	public static <D extends java.lang.Comparable<D>> Check<D> betweenInclusive(D left, D right) {
		return ComparisonChecks.betweenInclusive(left, right);
	}

	/**
	 * @see StringChecks#equalCaseInsensitiveTo(String)
	 */
	public static Check<java.lang.String> equalCaseInsensitiveTo(String expectedValue) {
		return StringChecks.equalCaseInsensitiveTo(expectedValue);
	}

	/**
	 * @see StringChecks#emptyString()
	 */
	public static Check<java.lang.String> emptyString() {
		return StringChecks.emptyString();
	}

	/**
	 * @see StringChecks#trim(Check<? super java.lang.String>)
	 */
	public static Check<java.lang.String> trim(Check<? super java.lang.String> check) {
		return StringChecks.trim(check);
	}

	/**
	 * @see StringChecks#startsWith(String)
	 */
	public static Check<java.lang.String> startsWith(String prefix) {
		return StringChecks.startsWith(prefix);
	}

	/**
	 * @see StringChecks#startsCaseInsensitiveWith(String)
	 */
	public static Check<java.lang.String> startsCaseInsensitiveWith(String prefix) {
		return StringChecks.startsCaseInsensitiveWith(prefix);
	}

	/**
	 * @see StringChecks#endsWith(String)
	 */
	public static Check<java.lang.String> endsWith(String suffix) {
		return StringChecks.endsWith(suffix);
	}

	/**
	 * @see StringChecks#endsCaseInsensitiveWith(String)
	 */
	public static Check<java.lang.String> endsCaseInsensitiveWith(String suffix) {
		return StringChecks.endsCaseInsensitiveWith(suffix);
	}

	/**
	 * @see StringChecks#contains(String)
	 */
	public static Check<java.lang.String> contains(String substring) {
		return StringChecks.contains(substring);
	}

	/**
	 * @see StringChecks#containsCaseInsensitive(String)
	 */
	public static Check<java.lang.String> containsCaseInsensitive(String substring) {
		return StringChecks.containsCaseInsensitive(substring);
	}

	/**
	 * @see StringChecks#matches(java.util.regex.Pattern)
	 */
	public static Check<java.lang.String> matches(java.util.regex.Pattern pattern) {
		return StringChecks.matches(pattern);
	}

	/**
	 * @see StringChecks#matchesPattern(String)
	 */
	public static Check<java.lang.String> matchesPattern(String pattern) {
		return StringChecks.matchesPattern(pattern);
	}

	/**
	 * @see XmlChecks#matchesXPath(String)
	 */
	public static Check<java.lang.Object> matchesXPath(String xPath) {
		return XmlChecks.matchesXPath(xPath);
	}

	/**
	 * @see XmlChecks#hasNode(String, javax.xml.namespace.QName)
	 */
	public static TransformationBuilder<java.lang.Object,fluent.validation.CheckBuilder<java.lang.Object>> hasNode(String xPath, javax.xml.namespace.QName type) {
		return XmlChecks.hasNode(xPath, type);
	}

	/**
	 * @see XmlChecks#hasNode(String)
	 */
	public static TransformationBuilder<java.lang.Object,fluent.validation.CheckBuilder<java.lang.Object>> hasNode(String xPath) {
		return XmlChecks.hasNode(xPath);
	}

	/**
	 * @see XmlChecks#hasTextContent(String)
	 */
	public static TransformationBuilder<java.lang.Object,fluent.validation.CheckBuilder<java.lang.String>> hasTextContent(String xPath) {
		return XmlChecks.hasTextContent(xPath);
	}

	/**
	 * Use string parameter as XML content, and check the file using DOM Document check.
	 *
	 * @param check DOM Document check.
	 * @return String content check.
	 * @see XmlChecks#parseXml(Check<? super org.w3c.dom.Document>)
	 */
	public static Check<java.lang.String> parseXml(Check<? super org.w3c.dom.Document> check) {
		return XmlChecks.parseXml(check);
	}

	/**
	 * Use string parameter as path to XML file, and check the file using DOM Document check.
	 *
	 * @param check DOM Document check.
	 * @return File name check.
	 * @see XmlChecks#loadXml(Check<? super org.w3c.dom.Document>)
	 */
	public static Check<java.lang.String> loadXml(Check<? super org.w3c.dom.Document> check) {
		return XmlChecks.loadXml(check);
	}

}
