/*
 * Copyright © 2018 Ondrej Fischer. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that
 * the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
 *    following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other materials provided with the distribution.
 *
 * 3. The name of the author may not be used to endorse or promote products derived from this software without
 *     specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY [LICENSOR] "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT
 * NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package fluent.validation;

import java.util.*;
import java.util.concurrent.BlockingQueue;

import static java.util.concurrent.TimeUnit.MILLISECONDS;

public class Functions {

    @SafeVarargs
    public static <D> Set<D> setOf(D... values) {
        switch (values.length) {
            case 0:
                return Collections.emptySet();
            case 1:
                return Collections.singleton(values[0]);
            default:
                return Collections.unmodifiableSet(new LinkedHashSet<>(Arrays.asList(values)));
        }
    }

    public static Set<String> splitToSet(String value, String splitter) {
        return setOf(value.split(splitter));
    }

    public static <D> Iterator<D> iterator(Iterable<D> iterable) {
        return iterable.iterator();
    }

    public static <D> Iterator<D> queueIterator(Queue<D> queue) {
        return new Iterator<D>() {
            @Override
            public boolean hasNext() {
                return !queue.isEmpty();
            }

            @Override
            public D next() {
                return queue.poll();
            }
        };
    }

    private static <D> D poll(BlockingQueue<D> queue, long timeout) {
        try {
            return queue.poll(timeout, MILLISECONDS);
        } catch (InterruptedException e) {
            throw new UncheckedInterruptedException("", e);
        }
    }

    public static <D> Iterator<D> blockingQueueIterator(BlockingQueue<D> queue, long timeout) {
        return new Iterator<D>() {
            private D data = poll(queue, timeout);
            @Override
            public boolean hasNext() {
                return data != null;
            }

            @Override
            public D next() {
                D last = data;
                data = poll(queue, timeout);
                return last;
            }
        };
    }

}
