package flexible.xd.android_base.utils;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.ActivityManager;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.pm.ResolveInfo;
import android.content.pm.Signature;
import android.graphics.drawable.Drawable;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.net.Uri;
import android.os.CountDownTimer;
import android.util.Log;
import android.view.View;
import android.view.ViewGroup;
import android.view.inputmethod.InputMethodManager;
import android.webkit.WebChromeClient;
import android.webkit.WebSettings;
import android.webkit.WebView;
import android.webkit.WebViewClient;
import android.widget.EditText;
import android.widget.ListAdapter;
import android.widget.ListView;
import android.widget.ProgressBar;
import android.widget.RelativeLayout;
import android.widget.TextView;

import java.io.File;
import java.util.ArrayList;
import java.util.List;




/**
 * <pre>
 *     author: Blankj
 *     blog  : http://blankj.com
 *     time  : 2016/08/02
 *     desc  : App 相关工具类
 * </pre>
 */
public class AppUtil {




        /**
         * 判断 App 是否安装
         *
         * @param action   action
         * @param category category
         * @return {@code true}: 已安装<br>{@code false}: 未安装
         */
        public static boolean isInstallApp(final String action, final String category) {
            Intent intent = new Intent(action);
            intent.addCategory(category);
            PackageManager pm = Utils.getApp().getPackageManager();
            ResolveInfo info = pm.resolveActivity(intent, 0);
            return info != null;
        }

        /**
         * 判断 App 是否安装
         *
         * @param packageName 包名
         * @return {@code true}: 已安装<br>{@code false}: 未安装
         */
        public static boolean isInstallApp(final String packageName) {
            return !isSpace(packageName) && IntentUtils.getLaunchAppIntent(packageName) != null;
        }

        /**
         * 安装 App(支持 8.0)
         * <p>8.0 需添加权限
         * {@code <uses-permission android:name="android.permission.REQUEST_INSTALL_PACKAGES" />}</p>
         *
         * @param filePath  文件路径
         * @param authority 7.0 及以上安装需要传入清单文件中的{@code <provider>}的 authorities 属性
         *                  <br>参看 https://developer.android.com/reference/android/support/v4/content/FileProvider.html
         */
        public static void installApp(final String filePath, final String authority) {
            installApp(FileUtils.getFileByPath(filePath), authority);
        }

        /**
         * 安装 App（支持 8.0）
         * <p>8.0 需添加权限
         * {@code <uses-permission android:name="android.permission.REQUEST_INSTALL_PACKAGES" />}</p>
         *
         * @param file      文件
         * @param authority 7.0 及以上安装需要传入清单文件中的{@code <provider>}的 authorities 属性
         *                  <br>参看 https://developer.android.com/reference/android/support/v4/content/FileProvider.html
         */
        public static void installApp(final File file, final String authority) {
            if (!FileUtils.isFileExists(file)) return;
            Utils.getApp().startActivity(IntentUtils.getInstallAppIntent(file, authority, true));
        }

        /**
         * 安装 App（支持 8.0）
         * <p>8.0 需添加权限
         * {@code <uses-permission android:name="android.permission.REQUEST_INSTALL_PACKAGES" />}</p>
         *
         * @param activity    activity
         * @param filePath    文件路径
         * @param authority   7.0 及以上安装需要传入清单文件中的{@code <provider>}的 authorities 属性
         *                    <br>参看 https://developer.android.com/reference/android/support/v4/content/FileProvider.html
         * @param requestCode 请求值
         */
        public static void installApp(final Activity activity,
                                      final String filePath,
                                      final String authority,
                                      final int requestCode) {
            installApp(activity, FileUtils.getFileByPath(filePath), authority, requestCode);
        }

        /**
         * 安装 App（支持 8.0）
         * <p>8.0 需添加权限
         * {@code <uses-permission android:name="android.permission.REQUEST_INSTALL_PACKAGES" />}</p>
         *
         * @param activity    activity
         * @param file        文件
         * @param authority   7.0 及以上安装需要传入清单文件中的{@code <provider>}的 authorities 属性
         *                    <br>参看 https://developer.android.com/reference/android/support/v4/content/FileProvider.html
         * @param requestCode 请求值
         */
        public static void installApp(final Activity activity,
                                      final File file,
                                      final String authority,
                                      final int requestCode) {
            if (!FileUtils.isFileExists(file)) return;
            activity.startActivityForResult(IntentUtils.getInstallAppIntent(file, authority),
                    requestCode);
        }

        /**
         * 静默安装 App
         * <p>非 root 需添加权限
         * {@code <uses-permission android:name="android.permission.INSTALL_PACKAGES" />}</p>
         *
         * @param filePath 文件路径
         * @return {@code true}: 安装成功<br>{@code false}: 安装失败
         */
        public static boolean installAppSilent(final String filePath) {
            File file = FileUtils.getFileByPath(filePath);
            if (!FileUtils.isFileExists(file)) return false;
            boolean isRoot = isDeviceRooted();
            String command = "LD_LIBRARY_PATH=/vendor/lib:/system/lib pm install " + filePath;
            ShellUtils.CommandResult commandResult = ShellUtils.execCmd(command, isRoot);
            if (commandResult.successMsg != null
                    && commandResult.successMsg.toLowerCase().contains("success")) {
                return true;
            } else {
                command = "LD_LIBRARY_PATH=/vendor/lib:/system/lib64 pm install " + filePath;
                commandResult = ShellUtils.execCmd(command, isRoot, true);
                return commandResult.successMsg != null
                        && commandResult.successMsg.toLowerCase().contains("success");
            }
        }

        /**
         * 卸载 App
         *
         * @param packageName 包名
         */
        public static void uninstallApp(final String packageName) {
            if (isSpace(packageName)) return;
            Utils.getApp().startActivity(IntentUtils.getUninstallAppIntent(packageName, true));
        }

        /**
         * 卸载 App
         *
         * @param activity    activity
         * @param packageName 包名
         * @param requestCode 请求值
         */
        public static void uninstallApp(final Activity activity,
                                        final String packageName,
                                        final int requestCode) {
            if (isSpace(packageName)) return;
            activity.startActivityForResult(IntentUtils.getUninstallAppIntent(packageName), requestCode);
        }

        /**
         * 静默卸载 App
         * <p>非 root 需添加权限
         * {@code <uses-permission android:name="android.permission.DELETE_PACKAGES" />}</p>
         *
         * @param packageName 包名
         * @param isKeepData  是否保留数据
         * @return {@code true}: 卸载成功<br>{@code false}: 卸载失败
         */
        public static boolean uninstallAppSilent(final String packageName, final boolean isKeepData) {
            if (isSpace(packageName)) return false;
            boolean isRoot = isDeviceRooted();
            String command = "LD_LIBRARY_PATH=/vendor/lib:/system/lib pm uninstall "
                    + (isKeepData ? "-k " : "")
                    + packageName;
            ShellUtils.CommandResult commandResult = ShellUtils.execCmd(command, isRoot, true);
            if (commandResult.successMsg != null
                    && commandResult.successMsg.toLowerCase().contains("success")) {
                return true;
            } else {
                command = "LD_LIBRARY_PATH=/vendor/lib:/system/lib64 pm uninstall "
                        + (isKeepData ? "-k " : "")
                        + packageName;
                commandResult = ShellUtils.execCmd(command, isRoot, true);
                return commandResult.successMsg != null
                        && commandResult.successMsg.toLowerCase().contains("success");
            }
        }

        /**
         * 判断 App 是否有 root 权限
         *
         * @return {@code true}: 是<br>{@code false}: 否
         */
        public static boolean isAppRoot() {
            ShellUtils.CommandResult result = ShellUtils.execCmd("echo root", true);
            if (result.result == 0) return true;
            if (result.errorMsg != null) {
                Log.d("AppUtils", "isAppRoot() called" + result.errorMsg);
            }
            return false;
        }

        /**
         * 打开 App
         *
         * @param packageName 包名
         */
        public static void launchApp(final String packageName) {
            if (isSpace(packageName)) return;
            Utils.getApp().startActivity(IntentUtils.getLaunchAppIntent(packageName, true));
        }

        /**
         * 打开 App
         *
         * @param activity    activity
         * @param packageName 包名
         * @param requestCode 请求值
         */
        public static void launchApp(final Activity activity,
                                     final String packageName,
                                     final int requestCode) {
            if (isSpace(packageName)) return;
            activity.startActivityForResult(IntentUtils.getLaunchAppIntent(packageName), requestCode);
        }

        /**
         * 关闭 App
         */
        public static void exitApp() {
            List<Activity> activityList = Utils.sActivityList;
            for (int i = activityList.size() - 1; i >= 0; --i) {
                activityList.get(i).finish();
                activityList.remove(i);
            }
            System.exit(0);
        }

        /**
         * 获取 App 包名
         *
         * @return App 包名
         */
        public static String getAppPackageName() {
            return Utils.getApp().getPackageName();
        }

        /**
         * 获取 App 具体设置
         */
        public static void getAppDetailsSettings() {
            getAppDetailsSettings(Utils.getApp().getPackageName());
        }

        /**
         * 获取 App 具体设置
         *
         * @param packageName 包名
         */
        public static void getAppDetailsSettings(final String packageName) {
            if (isSpace(packageName)) return;
            Utils.getApp().startActivity(IntentUtils.getAppDetailsSettingsIntent(packageName, true));
        }

        /**
         * 获取 App 名称
         *
         * @return App 名称
         */
        public static String getAppName() {
            return getAppName(Utils.getApp().getPackageName());
        }

        /**
         * 获取 App 名称
         *
         * @param packageName 包名
         * @return App 名称
         */
        public static String getAppName(final String packageName) {
            if (isSpace(packageName)) return null;
            try {
                PackageManager pm = Utils.getApp().getPackageManager();
                PackageInfo pi = pm.getPackageInfo(packageName, 0);
                return pi == null ? null : pi.applicationInfo.loadLabel(pm).toString();
            } catch (PackageManager.NameNotFoundException e) {
                e.printStackTrace();
                return null;
            }
        }

        /**
         * 获取 App 图标
         *
         * @return App 图标
         */
        public static Drawable getAppIcon() {
            return getAppIcon(Utils.getApp().getPackageName());
        }

        /**
         * 获取 App 图标
         *
         * @param packageName 包名
         * @return App 图标
         */
        public static Drawable getAppIcon(final String packageName) {
            if (isSpace(packageName)) return null;
            try {
                PackageManager pm = Utils.getApp().getPackageManager();
                PackageInfo pi = pm.getPackageInfo(packageName, 0);
                return pi == null ? null : pi.applicationInfo.loadIcon(pm);
            } catch (PackageManager.NameNotFoundException e) {
                e.printStackTrace();
                return null;
            }
        }

        /**
         * 获取 App 路径
         *
         * @return App 路径
         */
        public static String getAppPath() {
            return getAppPath(Utils.getApp().getPackageName());
        }

        /**
         * 获取 App 路径
         *
         * @param packageName 包名
         * @return App 路径
         */
        public static String getAppPath(final String packageName) {
            if (isSpace(packageName)) return null;
            try {
                PackageManager pm = Utils.getApp().getPackageManager();
                PackageInfo pi = pm.getPackageInfo(packageName, 0);
                return pi == null ? null : pi.applicationInfo.sourceDir;
            } catch (PackageManager.NameNotFoundException e) {
                e.printStackTrace();
                return null;
            }
        }

        /**
         * 获取 App 版本号
         *
         * @return App 版本号
         */
        public static String getAppVersionName() {
            return getAppVersionName(Utils.getApp().getPackageName());
        }

        /**
         * 获取 App 版本号
         *
         * @param packageName 包名
         * @return App 版本号
         */
        public static String getAppVersionName(final String packageName) {
            if (isSpace(packageName)) return null;
            try {
                PackageManager pm = Utils.getApp().getPackageManager();
                PackageInfo pi = pm.getPackageInfo(packageName, 0);
                return pi == null ? null : pi.versionName;
            } catch (PackageManager.NameNotFoundException e) {
                e.printStackTrace();
                return null;
            }
        }

        /**
         * 获取 App 版本码
         *
         * @return App 版本码
         */
        public static int getAppVersionCode() {
            return getAppVersionCode(Utils.getApp().getPackageName());
        }

        /**
         * 获取 App 版本码
         *
         * @param packageName 包名
         * @return App 版本码
         */
        public static int getAppVersionCode(final String packageName) {
            if (isSpace(packageName)) return -1;
            try {
                PackageManager pm = Utils.getApp().getPackageManager();
                PackageInfo pi = pm.getPackageInfo(packageName, 0);
                return pi == null ? -1 : pi.versionCode;
            } catch (PackageManager.NameNotFoundException e) {
                e.printStackTrace();
                return -1;
            }
        }

        /**
         * 判断 App 是否是系统应用
         *
         * @return {@code true}: 是<br>{@code false}: 否
         */
        public static boolean isSystemApp() {
            return isSystemApp(Utils.getApp().getPackageName());
        }

        /**
         * 判断 App 是否是系统应用
         *
         * @param packageName 包名
         * @return {@code true}: 是<br>{@code false}: 否
         */
        public static boolean isSystemApp(final String packageName) {
            if (isSpace(packageName)) return false;
            try {
                PackageManager pm = Utils.getApp().getPackageManager();
                ApplicationInfo ai = pm.getApplicationInfo(packageName, 0);
                return ai != null && (ai.flags & ApplicationInfo.FLAG_SYSTEM) != 0;
            } catch (PackageManager.NameNotFoundException e) {
                e.printStackTrace();
                return false;
            }
        }

        /**
         * 判断 App 是否是 Debug 版本
         *
         * @return {@code true}: 是<br>{@code false}: 否
         */
        public static boolean isAppDebug() {
            return isAppDebug(Utils.getApp().getPackageName());
        }

        /**
         * 判断 App 是否是 Debug 版本
         *
         * @param packageName 包名
         * @return {@code true}: 是<br>{@code false}: 否
         */
        public static boolean isAppDebug(final String packageName) {
            if (isSpace(packageName)) return false;
            try {
                PackageManager pm = Utils.getApp().getPackageManager();
                ApplicationInfo ai = pm.getApplicationInfo(packageName, 0);
                return ai != null && (ai.flags & ApplicationInfo.FLAG_DEBUGGABLE) != 0;
            } catch (PackageManager.NameNotFoundException e) {
                e.printStackTrace();
                return false;
            }
        }

        /**
         * 获取 App 签名
         *
         * @return App 签名
         */
        public static Signature[] getAppSignature() {
            return getAppSignature(Utils.getApp().getPackageName());
        }

        /**
         * 获取 App 签名
         *
         * @param packageName 包名
         * @return App 签名
         */
        public static Signature[] getAppSignature(final String packageName) {
            if (isSpace(packageName)) return null;
            try {
                PackageManager pm = Utils.getApp().getPackageManager();
                @SuppressLint("PackageManagerGetSignatures")
                PackageInfo pi = pm.getPackageInfo(packageName, PackageManager.GET_SIGNATURES);
                return pi == null ? null : pi.signatures;
            } catch (PackageManager.NameNotFoundException e) {
                e.printStackTrace();
                return null;
            }
        }

        /**
         * 获取应用签名的的 SHA1 值
         * <p>可据此判断高德，百度地图 key 是否正确</p>
         *
         * @return 应用签名的 SHA1 字符串, 比如：53:FD:54:DC:19:0F:11:AC:B5:22:9E:F1:1A:68:88:1B:8B:E8:54:42
         */
        public static String getAppSignatureSHA1() {
            return getAppSignatureSHA1(Utils.getApp().getPackageName());
        }

        /**
         * 获取应用签名的的 SHA1 值
         * <p>可据此判断高德，百度地图 key 是否正确</p>
         *
         * @param packageName 包名
         * @return 应用签名的 SHA1 字符串, 比如：53:FD:54:DC:19:0F:11:AC:B5:22:9E:F1:1A:68:88:1B:8B:E8:54:42
         */
        public static String getAppSignatureSHA1(final String packageName) {
            Signature[] signature = getAppSignature(packageName);
            if (signature == null) return null;
            return EncryptUtils.encryptSHA1ToString(signature[0].toByteArray()).
                    replaceAll("(?<=[0-9A-F]{2})[0-9A-F]{2}", ":$0");
        }

        /**
         * 判断 App 是否处于前台
         *
         * @return {@code true}: 是<br>{@code false}: 否
         */
        public static boolean isAppForeground() {
            ActivityManager manager =
                    (ActivityManager) Utils.getApp().getSystemService(Context.ACTIVITY_SERVICE);
            List<ActivityManager.RunningAppProcessInfo> info = manager.getRunningAppProcesses();
            if (info == null || info.size() == 0) return false;
            for (ActivityManager.RunningAppProcessInfo aInfo : info) {
                if (aInfo.importance == ActivityManager.RunningAppProcessInfo.IMPORTANCE_FOREGROUND) {
                    return aInfo.processName.equals(Utils.getApp().getPackageName());
                }
            }
            return false;
        }

        /**
         * 判断 App 是否处于前台
         * <p>当不是查看当前 App，且 SDK 大于 21 时，
         * 需添加权限 {@code <uses-permission android:name="android.permission.PACKAGE_USAGE_STATS" />}</p>
         *
         * @param packageName 包名
         * @return {@code true}: 是<br>{@code false}: 否
         */
        public static boolean isAppForeground(final String packageName) {
            return !isSpace(packageName) && packageName.equals(ProcessUtils.getForegroundProcessName());
        }

        /**
         * 封装 App 信息的 Bean 类
         */
        public static class AppInfo {

            private String   name;
            private Drawable icon;
            private String   packageName;
            private String   packagePath;
            private String   versionName;
            private int      versionCode;
            private boolean  isSystem;

            public Drawable getIcon() {
                return icon;
            }

            public void setIcon(final Drawable icon) {
                this.icon = icon;
            }

            public boolean isSystem() {
                return isSystem;
            }

            public void setSystem(final boolean isSystem) {
                this.isSystem = isSystem;
            }

            public String getName() {
                return name;
            }

            public void setName(final String name) {
                this.name = name;
            }

            public String getPackageName() {
                return packageName;
            }

            public void setPackageName(final String packageName) {
                this.packageName = packageName;
            }

            public String getPackagePath() {
                return packagePath;
            }

            public void setPackagePath(final String packagePath) {
                this.packagePath = packagePath;
            }

            public int getVersionCode() {
                return versionCode;
            }

            public void setVersionCode(final int versionCode) {
                this.versionCode = versionCode;
            }

            public String getVersionName() {
                return versionName;
            }

            public void setVersionName(final String versionName) {
                this.versionName = versionName;
            }

            /**
             * @param name        名称
             * @param icon        图标
             * @param packageName 包名
             * @param packagePath 包路径
             * @param versionName 版本号
             * @param versionCode 版本码
             * @param isSystem    是否系统应用
             */
            public AppInfo(String packageName, String name, Drawable icon, String packagePath,
                           String versionName, int versionCode, boolean isSystem) {
                this.setName(name);
                this.setIcon(icon);
                this.setPackageName(packageName);
                this.setPackagePath(packagePath);
                this.setVersionName(versionName);
                this.setVersionCode(versionCode);
                this.setSystem(isSystem);
            }

            @Override
            public String toString() {
                return "pkg name: " + getPackageName() +
                        "\napp name: " + getName() +
                        "\napp path: " + getPackagePath() +
                        "\napp v name: " + getVersionName() +
                        "\napp v code: " + getVersionCode() +
                        "\nis system: " + isSystem();
            }
        }

        /**
         * 获取 App 信息
         * <p>AppInfo（名称，图标，包名，版本号，版本 Code，是否系统应用）</p>
         *
         * @return 当前应用的 AppInfo
         */
        public static AppInfo getAppInfo() {
            return getAppInfo(Utils.getApp().getPackageName());
        }

        /**
         * 获取 App 信息
         * <p>AppInfo（名称，图标，包名，版本号，版本 Code，是否系统应用）</p>
         *
         * @param packageName 包名
         * @return 当前应用的 AppInfo
         */
        public static AppInfo getAppInfo(final String packageName) {
            try {
                PackageManager pm = Utils.getApp().getPackageManager();
                PackageInfo pi = pm.getPackageInfo(packageName, 0);
                return getBean(pm, pi);
            } catch (PackageManager.NameNotFoundException e) {
                e.printStackTrace();
                return null;
            }
        }

        /**
         * 得到 AppInfo 的 Bean
         *
         * @param pm 包的管理
         * @param pi 包的信息
         * @return AppInfo 类
         */
        private static AppInfo getBean(final PackageManager pm, final PackageInfo pi) {
            if (pm == null || pi == null) return null;
            ApplicationInfo ai = pi.applicationInfo;
            String packageName = pi.packageName;
            String name = ai.loadLabel(pm).toString();
            Drawable icon = ai.loadIcon(pm);
            String packagePath = ai.sourceDir;
            String versionName = pi.versionName;
            int versionCode = pi.versionCode;
            boolean isSystem = (ApplicationInfo.FLAG_SYSTEM & ai.flags) != 0;
            return new AppInfo(packageName, name, icon, packagePath, versionName, versionCode, isSystem);
        }

        /**
         * 获取所有已安装 App 信息
         * <p>{@link #getBean(PackageManager, PackageInfo)}
         * （名称，图标，包名，包路径，版本号，版本 Code，是否系统应用）</p>
         * <p>依赖上面的 getBean 方法</p>
         *
         * @return 所有已安装的 AppInfo 列表
         */
        public static List<AppInfo> getAppsInfo() {
            List<AppInfo> list = new ArrayList<>();
            PackageManager pm = Utils.getApp().getPackageManager();
            // 获取系统中安装的所有软件信息
            List<PackageInfo> installedPackages = pm.getInstalledPackages(0);
            for (PackageInfo pi : installedPackages) {
                AppInfo ai = getBean(pm, pi);
                if (ai == null) continue;
                list.add(ai);
            }
            return list;
        }

        /**
         * 清除 App 所有数据
         *
         * @param dirPaths 目录路径
         * @return {@code true}: 成功<br>{@code false}: 失败
         */
        public static boolean cleanAppData(final String... dirPaths) {
            File[] dirs = new File[dirPaths.length];
            int i = 0;
            for (String dirPath : dirPaths) {
                dirs[i++] = new File(dirPath);
            }
            return cleanAppData(dirs);
        }

        /**
         * 清除 App 所有数据
         *
         * @param dirs 目录
         * @return {@code true}: 成功<br>{@code false}: 失败
         */
        public static boolean cleanAppData(final File... dirs) {
            boolean isSuccess = CleanUtils.cleanInternalCache();
            isSuccess &= CleanUtils.cleanInternalDbs();
            isSuccess &= CleanUtils.cleanInternalSp();
            isSuccess &= CleanUtils.cleanInternalFiles();
            isSuccess &= CleanUtils.cleanExternalCache();
            for (File dir : dirs) {
                isSuccess &= CleanUtils.cleanCustomCache(dir);
            }
            return isSuccess;
        }

        private static boolean isSpace(final String s) {
            if (s == null) return true;
            for (int i = 0, len = s.length(); i < len; ++i) {
                if (!Character.isWhitespace(s.charAt(i))) {
                    return false;
                }
            }
            return true;
        }

        private static boolean isDeviceRooted() {
            String su = "su";
            String[] locations = {"/system/bin/", "/system/xbin/", "/sbin/", "/system/sd/xbin/",
                    "/system/bin/failsafe/", "/data/local/xbin/", "/data/local/bin/", "/data/local/"};
            for (String location : locations) {
                if (new File(location + su).exists()) {
                    return true;
                }
            }
            return false;
        }




    /**
     * 解决lv与sv嵌套  lv长度不正确
     *
     * @param listView
     */

    public static void setListViewHeightBasedOnChildren(ListView listView) {
        ListAdapter listAdapter = listView.getAdapter();
        if (listAdapter == null) {
            return;
        }

        int totalHeight = 0;
        for (int i = 0; i < listAdapter.getCount(); i++) {
            View listItem = listAdapter.getView(i, null, listView);
            listItem.measure(0, 0);
            totalHeight += listItem.getMeasuredHeight();
        }

        ViewGroup.LayoutParams params = listView.getLayoutParams();
        params.height = totalHeight + (listView.getDividerHeight() * (listAdapter.getCount() - 1));
        ((ViewGroup.MarginLayoutParams) params).setMargins(15, 15, 15, 15);
        listView.setLayoutParams(params);

    }

    /**
     * 隐藏et键盘
     *
     * @param edittext
     */
    public static void HideEditKeyBoard(EditText edittext, Context ctx) {
        InputMethodManager imm = (InputMethodManager) ctx
                .getSystemService(Context.INPUT_METHOD_SERVICE);
        imm.hideSoftInputFromWindow(edittext.getWindowToken(), 0);
    }

    /**
     * 设置wv的一些通用属性
     */

    public static void initWebView(final WebView wv, final Activity ctx) {
        WebSettings webSettings = wv.getSettings();
        webSettings.setTextSize(WebSettings.TextSize.NORMAL);//设置字体大小
        webSettings.setUseWideViewPort(false);//设置此属性，可任意比例缩放
        webSettings.setLoadWithOverviewMode(true);
        webSettings.setUseWideViewPort(true); //将图片调整到适合webview的大小
        webSettings.setJavaScriptEnabled(true);
        webSettings.setBuiltInZoomControls(false);
        webSettings.setLayoutAlgorithm(WebSettings.LayoutAlgorithm.SINGLE_COLUMN); //支持内容重新布局
        webSettings.setSupportZoom(false);
        wv.setWebViewClient(new WebViewClient() {


            @Override
            public boolean shouldOverrideUrlLoading(WebView view, String url) {
                view.loadUrl(url);
                return true;
            }
        });

        wv.setOnLongClickListener(new View.OnLongClickListener() {
            @Override
            public boolean onLongClick(View view) {
                return true;
            }
        });
        wv.setWebChromeClient(new WebChromeClient() {


            private ProgressBar mProgressbar;

            public void onProgressChanged(WebView view, int progress) {
                if (mProgressbar == null) {
                    mProgressbar = new ProgressBar(ctx, null, android.R.attr.progressBarStyleLarge);
                    RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(200, 200);
                    RelativeLayout rv = new RelativeLayout(ctx);
                    params.addRule(RelativeLayout.CENTER_IN_PARENT);
                    rv.addView(mProgressbar, params);
                    RelativeLayout.LayoutParams params1 = new RelativeLayout.LayoutParams(RelativeLayout.LayoutParams.MATCH_PARENT, RelativeLayout.LayoutParams.MATCH_PARENT);
                    wv.addView(rv, params1);
                }
                if (progress == 100) {
                    mProgressbar.setVisibility(View.GONE);
                } else {
                    mProgressbar.setVisibility(View.VISIBLE);
                }
            }

            @Override
            public void onReceivedTitle(WebView view, String title) {
                super.onReceivedTitle(view, title);
//                ctx.setTitle(title);
            }
        });

    }



    /**
     * 倒计时
     *
     * @param txtTime
     * @param AllTime
     * @param JianGe
     */
    public static <T extends TextView> CountDownTimer timerSeckill(final T txtTime, long AllTime, final int JianGe, final int code) {
        if (code == 0) {
            txtTime.setClickable(false);
        }
        CountDownTimer timer = new CountDownTimer(AllTime, JianGe) {

            @Override
            public void onTick(long millisUntilFinished) {
                if (code == 0) {
                    txtTime.setText(String.valueOf(millisUntilFinished / JianGe) + "s后重新获取");
                } else {
                    txtTime.setText("跳过  " + String.valueOf(millisUntilFinished / JianGe) + "s");
                }
            }
            @Override
            public void onFinish() {

                if (code == 0) {
                    txtTime.setText("获取验证码");
                    txtTime.setClickable(true);
                } else {
                    txtTime.setText("跳过  0s");
                }
            }
        };
        return timer.start();
    }

}
