package digital.tail.sdk.tail_mobile_sdk;

import android.app.job.JobInfo;
import android.app.job.JobScheduler;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.os.AsyncTask;
import android.os.Build;
import android.os.PersistableBundle;
import android.renderscript.Sampler;
import android.text.format.DateFormat;
import android.util.Log;
import android.view.Gravity;
import android.widget.Toast;

import com.google.android.gms.security.ProviderInstaller;
import com.google.gson.Gson;
import com.google.gson.JsonSyntaxException;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.math.BigInteger;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import digital.tail.sdk.tail_mobile_sdk.exception.TailDMPException;

/**
 * Created by babadopulos on 08/02/17.
 */

public class TailDMP implements ProviderInstaller.ProviderInstallListener{
    public final static String TAG = TailDMPValues.TAG;

    //SDK version full always add (1.2.1, 1.2.3, 1.2.5 ... )
    public final static String VERSION = "1.2.21";

    // Tail DMP singleton instance
    private static TailDMP INSTANCE;

    public TailDMP_Config config;
    public TailDMP_Crypto tailDMPCrypto;

    public SharedPreferences preferences;
    public Context context;

    public TailDMPDeviceMapping deviceMapping;
    public TailDMP_Config tailDMP_config = new TailDMP_Config();

    public TailDMPValues dmpValues = null;

    //is user optin
    private boolean optin = false;

    //change the url to send data to sandbox, for debug
    private boolean enabledSandbox = false;

    //sends data only if we are using wifi connection
    private boolean sendDataOnWifiOnly = false;

    public String clientData = "";

    public TailDMPActivityTracker tracker = null;

     /*
     * Services Schedulers
     */

    private ComponentName tailServiceComponent;
    private ComponentName tailServiceComponentSendData;
    private ComponentName tailServiceComponentSendALLData;

    private TailDMPJobService tService;
    private TailDMPCollectDataJobService tCollectService;
    private TailDMPSendDataJobService tSendService;
    private TailDMPSendALLDataJobService tSendALLService;
    private JobInfo.Builder builder;
    public final static int JOB_ID = 9999;

    private JobInfo.Builder builder_sendData ;
    public final static int JOB_ID_SENDURL = 91553;

    private JobInfo.Builder builder_sendALLData ;
    public final static int JOB_ID_SENDALLURL = 155399;

    //prevents send a new Async task before other one finished its execution
    private boolean sendingDataDirect = false;

    private boolean securityPatchUpdated = false;


    /**
     * Return TailDMP instance.
     *
     * @return TailDMP instance
     * @throws TailDMPException
     */
    public static synchronized TailDMP getInstance() throws TailDMPException {
        if (TailDMP.INSTANCE == null) {
            throw new TailDMPException("TailDMP not initialized. Call TailDMP.initialize() first.");
        }
        return TailDMP.INSTANCE;
    }

    /**
     * Initialize TailDMP singleton.
     * Call it only once during application start up.
     *
     * @param context
     * @throws TailDMPException
     */
    public static synchronized void initialize(Context context) throws TailDMPException {
        if (TailDMP.INSTANCE != null) {
            throw new TailDMPException("TailDMP already initialized. Call TailDMP.initialize() only once.");
        }

        TailDMP.INSTANCE = new TailDMP(context);

    }

    /**
     * Exists only to prevent accidental instantiation
     */
    protected TailDMP() {

    }


    private TailDMP(Context context) throws TailDMPException {

        //save context to be used later
        this.context = context;

        //creates config
        this.config = new TailDMP_Config();

        //creates preferences
        this.preferences = context.getSharedPreferences(TailDMP_Config.NAMESPACE, Context.MODE_PRIVATE);

        //initialize the config
        this.tailDMP_config.initialize(context);
        //Log.i(TailDMPValues.TAG, "accountID : "+this.tailDMP_config.getConfig());

        //initialize crypto helper
        this.tailDMPCrypto = new TailDMP_Crypto();

        //creates the device mapping, we get all needed data to send to a webservice later
        this.deviceMapping = new TailDMPDeviceMapping(context);

        //adds accountId from config json file in assets
        this.deviceMapping.setAccountID(this.tailDMP_config.getAccountID());


        if (android.os.Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            //instantiate the scheduler service
            //this.tService = new TailDMPJobService();
            this.tCollectService = new TailDMPCollectDataJobService();
            this.tSendService = new TailDMPSendDataJobService();
            this.tSendALLService = new TailDMPSendALLDataJobService();
        }

        //initialize dbhelper to create db
        TailDMPDb.getInstance(context);

        //save data from TailDMPConfig.json on preferences
        loadCurrentSavedValues();

        //try to get the app name
        resolveAppPackageName();

        //try to load if sandbox is enabled
        loadSavedEnabledSandbox();

        //try to create Google api client
        tracker = new TailDMPActivityTracker(context,this.deviceMapping);

        Log.d(TAG, "++++++++++++++++++++++++++++");
        Log.d(TAG, "+ TAIL SDK VERSION: "+ VERSION);
        Log.d(TAG, "++++++++++++++++++++++++++++");



        if (android.os.Build.VERSION.SDK_INT < Build.VERSION_CODES.LOLLIPOP) {
            //try to call provider installer if api < 21 (5.0)
            // we need security provider updated to send data with TSLv1, TSLv1.1, etc
            // the result of this operation will be handled by onProviderInstalled and onProviderInstallFailed
            try{
                Log.i(TailDMPValues.TAG, "Checking if we have an updated Google Play Services ");
                ProviderInstaller.installIfNeededAsync(this.context, this);
            }catch (Exception e){
                e.printStackTrace();
            }

        }else{
            securityPatchUpdated = true;
        }



    }



    //if patch is allready instaled
    @Override
    public void onProviderInstalled() {
        securityPatchUpdated = true;
        Log.d(TailDMPValues.TAG, "Google Play Services patch installed");
    }

    //if  patch is not installed
    @Override
    public void onProviderInstallFailed(int i, Intent intent) {
        securityPatchUpdated = false;

        Log.e(TailDMPValues.TAG, "Fail to update Google Play Services ");
    }

    private void resolveAppPackageName(){
        final PackageManager pm = context.getApplicationContext().getPackageManager();
        ApplicationInfo ai;

        //trying to get the package name of a app using the SDK
        try {
            ai = pm.getApplicationInfo( context.getPackageName(), 0);
        } catch (final PackageManager.NameNotFoundException e) {
            ai = null;
        }

        //if we have Application info than we set the packagename
        if(ai == null){
            deviceMapping.setCurrentApp("unknown");
        }else{
            deviceMapping.setCurrentApp(ai.packageName);
        }
        //final String applicationName = (String) (ai != null ? pm.getApplicationLabel(ai) : "(unknown)");
        //Log.e(TailDMPValues.TAG, ai.packageName);
    }

    private void loadCurrentSavedValues() {
        //load data
        String savedValues = preferences.getString(TailDMP_Config.SAVED_VALUES, null);
        if (savedValues == null) {
            this.dmpValues = new TailDMPValues();
            return;
        }


        Gson gson = new Gson();

        try {
            this.dmpValues = gson.fromJson(savedValues, TailDMPValues.class);
        } catch (JsonSyntaxException e) {
            this.dmpValues = new TailDMPValues();
        }

    }

    private void saveCurrentValues() {
        String currentValues;

        if (this.dmpValues == null) {
            currentValues = null;
        } else {
            Gson gson = new Gson();
            currentValues = gson.toJson(this.dmpValues);
        }

        this.preferences.edit()
                .putString(TailDMP_Config.SAVED_VALUES, currentValues)
                .commit();

    }




    //checks if the tag that user is trying to send is valid
    private boolean checkTagValid(String tag){

        String tagPattern = "^[a-zA-Z0-9_.-=]*$";
        String cleanedTag = tag.trim();



        if(cleanedTag.isEmpty()){
            //Log.e(TailDMPValues.TAG,"INVALID TAG, Tag empty!");
            return false;
        }else if(cleanedTag.length() > 32){
            //Log.e(TAG,"BIGGER THAN 32 chars: " + cleanedTag.length());
            return false;
        }else{
            return cleanedTag.matches(tagPattern);
        }

    }

    public TailDMP addTags(String tag){

        String tgCleaned = tag.trim();

        if(checkTagValid(tgCleaned)){
            if(isOptin()){

                if(deviceMapping.hasMinimunData()){
                    deviceMapping.setTags(tgCleaned);
                }else{
                    Log.e(TailDMPValues.TAG,"You must have an accountID to add Tags");
                }

            }else{
                Log.e(TailDMPValues.TAG,"User must be optin to add tags ");
            }

        }else{
            Log.e(TailDMPValues.TAG,"INVALID TAG, Tags must not exceed 32 characters, can contain Numbers, Letters and the following special chars: - _ . = ");
        }

        return this;
    }

    /* DEPRICATED
    private ArrayList getTags(){

        return deviceMapping.getTags();
    }
    */


    /**
     *
     * Send the devicemapping data to server without a job scheduler
     *
     * @return
     */
    public TailDMP sendData(String tag ){

        if(securityPatchUpdated){

            //populate user with devicemap
            populateUserWDeviceMap();

            //stop any jobs that is running
            //stopJob();

            //prepare data to be sent
            String data = "";


            if(isOptin()){

                //check if we have the minimum data to be sent
                if(deviceMapping.hasMinimunData()){

                    boolean cansendData = false;

                    //clean whitespaces
                    String tgcleaned = tag.trim();

                    if(tgcleaned != null && tgcleaned.isEmpty()){
                        //this tag can be empty
                        cansendData = true;
                    }else if(tgcleaned.length()>0){
                        //if not empty
                        if(checkTagValid(tgcleaned)){
                            cansendData = true;
                        }else {
                            cansendData = false;
                        }

                    }else{
                        cansendData = false;

                    }


                        //init tracking activity
                        tracker.startActivityTracking();

                        //get geolocation data
                        deviceMapping.getGeoLocationData();

                        if(!sendingDataDirect){

                            //prevents send a new Async task before other one finished its execution
                            sendingDataDirect = true;

                            //Try to send data to server
                            TAsyncSendALLJSON t = new TAsyncSendALLJSON(context,deviceMapping){

                                @Override
                                protected void onPostExecute(String resultFromAsync){
                                    //set tags to devicemaps
                                    deviceMapping.setTagsStringToArrayList(resultFromAsync);
                                    sendingDataDirect = false;

                                }

                            };

                            if(cansendData){
                                t.sendtag = true;
                                t.tagToSend = tgcleaned;
                                t.execute();
                            }else{
                                sendingDataDirect = false;
                                Log.e(TailDMPValues.TAG,"INVALID TAG, Tag must not exceed 32 characters, can be an empty string, can contain Numbers, Letters and the following special chars: - _ . = ");
                                Log.e(TailDMPValues.TAG,"We can't send data with an invalid tag.");
                            }
                        }
                }else{
                    Log.e(TailDMPValues.TAG,"You can't send data to server without an accountID");
                }

            }else {
                Log.e(TailDMPValues.TAG,"To send data the User must be opt in");
            }

        //end of security patch IF ELSE
        }else{
            Log.e(TailDMPValues.TAG, "To send device's data you must update Google play services ");
        }


        return this;
    }

    /**
     *
     * Add period in minutes to service job execution
     *
     * minutesToCollect : Time interval to collect data of device
     *
     * minutesToSend : Time interval to send data to Tailtarget
     *
     * @param minutesToCollect
     * @param minutesToSend
     * @return
     */
    public TailDMP setIntervalToExecuteJob(int minutesToCollect, int minutesToSend){

        if(minutesToCollect <15){

            Log.e(TailDMP.TAG, "Warning, the interval to execute a job that collect data of device  must be at least 15 minutes to preserve device's battery. We are changing the value provided to 15 minutes");
            minutesToCollect = 15;
        }

        if (minutesToSend < minutesToCollect){
            Log.e(TailDMP.TAG, "Warning, the interval to execute a job that sends data of device to TailTarget must be bigger than minutesToCollect. We are changing the value provided to 60 minutes");
            minutesToSend = 60;
        }else if( minutesToSend > 1440){
            Log.e(TailDMP.TAG, "Warning, the interval to execute a job that sends data of device to TailTarget must be at most 1440 minutes/24hs . We are changing the value provided to 240 minutes/4hs ");
            minutesToSend = TailDMPDb.DB_VALUE_UPDATE_JOB_4HOURS;
        }

        //DEBUG
       //minutesToCollect = 1;
       //minutesToSend = 4;

        int time2Collect = minutesToCollect * 60000;
        int time2Send = minutesToSend * 60000;
        int time2trackActivity = time2Collect /2;


        //set time to collect data
        TailDMPDb db= TailDMPDb.getInstance(context);
        db.setUpdateJobTime(time2Collect,time2Send);

        //pass the period to tracker activity
        tracker.setTRACKING_PERIOD(time2trackActivity);

        return this;
    }



    private boolean checkDataExists(String data){
        if( data != null && data.length()>0 && !data.isEmpty()){
            return true;
        }else{
            return false;
        }

    }

    private void populateUserWDeviceMap(){

        Map dt2update = new HashMap<String,String>();
        if(checkDataExists(deviceMapping.getAccountID())){
            dt2update.put(TailDMPDb.DB_FIELD_ACCOUNTID,deviceMapping.getAccountID());
        }
        if(checkDataExists(deviceMapping.getApiVersion())){
            dt2update.put(TailDMPDb.DB_FIELD_APIVERSION,deviceMapping.getApiVersion());
        }
        if(checkDataExists(deviceMapping.getUserHash())){
            dt2update.put(TailDMPDb.DB_FIELD_USERHASH,deviceMapping.getUserHash());
        }

        if(checkDataExists(deviceMapping.getAdvertisingId())){
            dt2update.put(TailDMPDb.DB_FIELD_ADVERTISINGID,deviceMapping.getAdvertisingId());
        }
        if(checkDataExists(deviceMapping.getLanguage())){
            dt2update.put(TailDMPDb.DB_FIELD_LANGUAGE,deviceMapping.getLanguage());
        }
        if(checkDataExists(deviceMapping.getTimezone())){
            dt2update.put(TailDMPDb.DB_FIELD_TIMEZONE,deviceMapping.getTimezone());
        }
        if(checkDataExists(deviceMapping.getOs())){
            dt2update.put(TailDMPDb.DB_FIELD_OS,deviceMapping.getOs());
        }
        if(checkDataExists(deviceMapping.getBrand())){
            dt2update.put(TailDMPDb.DB_FIELD_BRAND,deviceMapping.getBrand());
        }
        if(checkDataExists(deviceMapping.getProduct())){
            dt2update.put(TailDMPDb.DB_FIELD_PRODUCT,deviceMapping.getProduct());
        }
        if(checkDataExists(deviceMapping.getDevice())){
            dt2update.put(TailDMPDb.DB_FIELD_DEVICE,deviceMapping.getDevice());
        }
        if(checkDataExists(deviceMapping.getHardware())){
            dt2update.put(TailDMPDb.DB_FIELD_HARDWARE,deviceMapping.getHardware());
        }
        if(checkDataExists(deviceMapping.getManufacturer())){
            dt2update.put(TailDMPDb.DB_FIELD_MANUFACTURER,deviceMapping.getManufacturer());
        }

        if(checkDataExists(deviceMapping.getModel())){
            dt2update.put(TailDMPDb.DB_FIELD_MODEL,deviceMapping.getModel());
        }

        if(checkDataExists(deviceMapping.getCarrierName())){
            dt2update.put(TailDMPDb.DB_FIELD_CARRIER_NAME,deviceMapping.getCarrierName());
        }
        if(checkDataExists(deviceMapping.getCurrentApp())){
            dt2update.put(TailDMPDb.DB_FIELD_CURRENTAPP,deviceMapping.getCurrentApp());
        }

        if(deviceMapping.getTags() != null && deviceMapping.getTags().size() >0){

            String listString = "";

            for (String s : deviceMapping.getTags())
            {
                listString += s ;
            }
            dt2update.put(TailDMPDb.DB_FIELD_TAGS,listString);
        }

        if(deviceMapping.getAllApps() != null && deviceMapping.getAllApps().size() >0){

            dt2update.put(TailDMPDb.DB_FIELD_APPS,deviceMapping.getAppsJSONFormated());

        }

        TailDMPDb db = TailDMPDb.getInstance(context);

        db.updateData(dt2update);

    }


    /**
     * Starts a scheduler job that sends data to a webservice according to period defined dy setIntervalToExecuteJob()
     *
     * @return
     */
    public  TailDMP startJob(){

        try {
            //stop any jobs that is running
            stopJob();

            //init tracking activity
            tracker.startActivityTracking();

            TailDMPDb db = TailDMPDb.getInstance(context);

            Map timeToUpdate = db.getUpdateJobTime();

            //if not defined, 15 min is the default (DB_VALUE_UPDATE_JOB_15MINUTES)
            int time2UpdateCollect = (int) timeToUpdate.get(TailDMPDb.DB_FIELD_UPDATE_COLLECT_JOB_TIME);

            // if not defined, 4 hours is teh default (DB_VALUE_UPDATE_JOB_4HOURS)
            int time2UpdateSend = (int) timeToUpdate.get(TailDMPDb.DB_FIELD_UPDATE_SEND_JOB_TIME);

            //populate user with devicemap
            populateUserWDeviceMap();


            if (android.os.Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                // only for Lollipop and newer versions

                //if optin start job
                if (isOptin()) {

                    //check if we have the minimum data to be sent
                    if (deviceMapping.hasMinimunData()) {

                        Log.d(TAG, "++++ STARTING JOBS! ++++");


                        //instantiate job class component
                        //tailServiceComponent = new ComponentName(context, TailDMPJobService.class);
                        tailServiceComponent = new ComponentName(context, TailDMPCollectDataJobService.class);
                        tailServiceComponentSendData = new ComponentName(context, TailDMPSendDataJobService.class);
                        tailServiceComponentSendALLData = new ComponentName(context, TailDMPSendALLDataJobService.class);

                        //create the scheduler to collect
                        builder = new JobInfo.Builder(JOB_ID, tailServiceComponent);

                        //create scheduler of sendURL
                        builder_sendData = new JobInfo.Builder(JOB_ID_SENDURL, tailServiceComponentSendData);

                        //create a scheduler to send all data 6/6 hours
                        builder_sendALLData = new JobInfo.Builder(JOB_ID_SENDALLURL, tailServiceComponentSendALLData);

                        // --> ANDROID 7
                        //We must handle schedule time in a different way for Android 7
                        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {

                            double flexmili = time2UpdateCollect * 0.1;

                            builder.setRequiresCharging(false);
                            //builder.setMinimumLatency(timeToUpdate); //run a job every setMinimumLatency(miliseconds)
                            builder.setPeriodic(time2UpdateCollect, (long) flexmili);
                            builder.setPersisted(true);



                            builder_sendData.setRequiresCharging(false);
                            //wait max 15 min of flex to send data
                            builder_sendData.setPeriodic(time2UpdateSend, (long) (3 * 60000));
                            builder_sendData.setPersisted(true);


                            builder_sendALLData.setRequiresCharging(false);
                            builder_sendALLData.setPeriodic(TailDMPDb.DB_VALUE_UPDATE_JOB_ALLDATA, (long) (3 * 60000)); // 6h/15min flex
                            builder_sendALLData.setPersisted(true);


                        } else {
                            // --> ANDROID < =6

                            //SERVICE TO COLLECT DATA
                            builder.setRequiresCharging(false);
                            builder.setPeriodic(time2UpdateCollect); //run a job every setPeriodic(miliseconds)
                            builder.setPersisted(true);
                            //builder.setRequiresDeviceIdle(true);
                            //builder.setRequiredNetworkType(JobInfo.NETWORK_TYPE_ANY);

                            // SERVICE TO SEND DIFFERENCES DATA
                            builder_sendData.setRequiresCharging(false);
                            builder_sendData.setPeriodic(time2UpdateSend); //run a job every setPeriodic(miliseconds)
                            builder_sendData.setPersisted(true);
                            //builder_sendData.setRequiredNetworkType(JobInfo.NETWORK_TYPE_ANY);

                            //SERVICE SEND ALL DATA
                            builder_sendALLData.setRequiresCharging(false);
                            builder_sendALLData.setPersisted(true);
                            //builder_sendALLData.setPeriodic(600000); //5 min
                            builder_sendALLData.setPeriodic(TailDMPDb.DB_VALUE_UPDATE_JOB_ALLDATA);
                            //builder_sendALLData.setRequiredNetworkType(JobInfo.NETWORK_TYPE_ANY);

                        }


                        //execute the jobs
                        tSendService.scheduleJob(builder_sendData.build(), context);
                        tCollectService.scheduleJob(builder.build(), context);
                        tSendALLService.scheduleJob(builder_sendALLData.build(), context);


                    } else {
                        Log.e(TailDMPValues.TAG, "You can't send data to server without an accountID");
                    }

                } else {
                    Log.e(TailDMPValues.TAG, "User must be optin (TailDMP.setOptin( true )to schedule a job, please ask him to allow us to do it!");
                }

            } else {
                Log.e(TailDMPValues.TAG, "Tail SDK Job Scheduler requires Android API 21 (LOLLIPOP) or higher.");
            }

        }catch (Exception e){
            e.printStackTrace();
        }




        return this;
    }

    // On android >=7 we must start specific jobs
    public TailDMP startJobWithID(int Jobid){

        if (android.os.Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {

            //init tracking activity
            tracker.startActivityTracking();

            TailDMPDb db = TailDMPDb.getInstance(context);

            Map timeToUpdate = db.getUpdateJobTime();

            int time2UpdateCollect = (int) timeToUpdate.get(TailDMPDb.DB_FIELD_UPDATE_COLLECT_JOB_TIME);
            int time2UpdateSend = (int) timeToUpdate.get(TailDMPDb.DB_FIELD_UPDATE_SEND_JOB_TIME);

            //populate user with devicemap
            populateUserWDeviceMap();

            //DEBUG
            //Log.i(TailDMP.TAG, "RESTART JOB ID"+ Jobid);

            //if optin start job
            if (isOptin()) {

                //check if we have the minimum data to be sent
                if (deviceMapping.hasMinimunData()) {

                    //instantiate job class component
                    //tailServiceComponent = new ComponentName(context, TailDMPJobService.class);
                    tailServiceComponent = new ComponentName(context, TailDMPCollectDataJobService.class);
                    tailServiceComponentSendData = new ComponentName(context, TailDMPSendDataJobService.class);
                    tailServiceComponentSendALLData = new ComponentName(context, TailDMPSendALLDataJobService.class);

                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {

                        //create the scheduler
                        builder = new JobInfo.Builder(JOB_ID, tailServiceComponent);

                        //create scheduler of sendURL
                        builder_sendData = new JobInfo.Builder(JOB_ID_SENDURL, tailServiceComponentSendData);

                        builder_sendALLData = new JobInfo.Builder(JOB_ID_SENDALLURL, tailServiceComponentSendALLData);


                        double flexmili = time2UpdateCollect * 0.1;


                        if (Jobid == 15953) {
                            //collect data

                            builder.setRequiresCharging(false);
                            //run a job every setMinimumLatency(miliseconds)
                            builder.setPeriodic(time2UpdateCollect, (long) flexmili);
                            builder.setPersisted(true);

                            //execute the jobs
                            tCollectService.scheduleJob(builder.build(), context);

                        } else if (Jobid == 155399) {
                            //sendALLdata
                            builder_sendALLData.setRequiresCharging(false);
                            builder_sendALLData.setPeriodic(TailDMPDb.DB_VALUE_UPDATE_JOB_ALLDATA, (long) (3 * 60000)); // 1h
                            builder_sendALLData.setPersisted(true);

                            tSendALLService.scheduleJob(builder_sendALLData.build(), context);

                        } else if (Jobid == 91553) {
                            // send data
                            builder_sendData.setRequiresCharging(false);
                            builder_sendData.setPeriodic(time2UpdateSend, (long) (3 * 60000));
                            builder_sendData.setPersisted(true);

                            tSendService.scheduleJob(builder_sendData.build(), context);

                        } else {
                            Log.e(TailDMP.TAG, "Can' find jobservice with jobID: " + Jobid + " to be schedulled");
                        }

                    }

                }
            }

        }else{
            Log.e(TailDMPValues.TAG, "Tail SDK Job Scheduler requires Android API 21 (LOLLIPOP) or higher.");
        }

        return this;
    }




    /**
     * Stops the service that sends data to a webservice
     *
     * @return
     */
    public TailDMP stopJob(){

        Log.d(TAG, "++++ STOP ALL JOBS! ++++");
       //Log.i(TailDMPValues.TAG,"TailDMP stopJob()");

        //run only if in android  >= 21
        if (android.os.Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            try{
                if (tCollectService != null) {

                    JobScheduler jobScheduler = (JobScheduler) context.getSystemService(Context.JOB_SCHEDULER_SERVICE);
                    jobScheduler.cancelAll();
                    //tService.stopJobService(context);
                    //tCollectService.stopJobService(context);
                    //tSendService.stopJobService(context);
                    //stop activity tracking
                    tracker.stopActivityTracking();
                }
            }catch(Exception e){
                e.printStackTrace();
            }


        }else{
            Log.i(TailDMPValues.TAG, "Tail SDK Job Scheduler requires Android API 21 (LOLLIPOP) or higher.");
        }
        return this;
    }


    /**
     * Returns client data from our servers
     *
     * @return
     */
    public TAsyncGetClientData getClientData(){



        //call async URLloader
        TAsyncGetClientData t = new TAsyncGetClientData(context);

        //returns the data sent by a server
        return t;

    }


    /**
     * Clear all defined variables from current user
     *
     * @return


    public TailDMP clearCurrentValues() {
        this.dmpValues = new TailDMPValues();
        saveCurrentValues();

        return this;
    }
    */

    /**
     * Define a SHA256 hash that identifies this user.
     * Must be 64 chars long.
     *
     * @param userHash
     * @return
     */
    /*
    public TailDMP setUserHash(String userHash) throws TailDMPException {

        if (userHash == null || userHash.length() != 64) {
            throw new TailDMPException("TailDMP: User hash must be SHA256 and 64 chars long");
        }

        return this;
    }
 */


    /**
     * Set/unset sandbox endpoint for debug purposes
    **/

    public void enableSandbox(boolean enable){
        enabledSandbox = enable;
        try {
            String val2save = (enabledSandbox) ? "1" : "0";
            //saves sandbox value to use later
            this.preferences.edit()
                    .putString(TailDMP_Config.SANDBOX_ENABLED, val2save)
                    .commit();
        }catch(Exception e){
            e.printStackTrace();
        }
        this.deviceMapping.sandboxEnabled = enable;

    }

    public boolean isEnabledSandbox(){


        return enabledSandbox;
    }

    private void loadSavedEnabledSandbox(){
        String enabled = null;

        try{
            enabled = preferences.getString(TailDMP_Config.SANDBOX_ENABLED, null);
        }catch(Exception e){
            e.printStackTrace();
        }

        if (enabled != null) {
            if(enabled.equals("1")){
                enabledSandbox = true;
            }else{
                enabledSandbox = false;
            }
        }

        this.deviceMapping.sandboxEnabled = enabledSandbox;
    }


    /**
     *
     * Set user data to be encrypted
     *
     *
      * @param emailToHash
     */
    public void generateUserHashFromEmail(String emailToHash)  {

        if(isValidEmail(emailToHash)){
            deviceMapping.setUserHashEmail(emailToHash);
        }else{
            Log.e(TailDMP.TAG, "Warning, Invalid format to generate a hash. Email must be all lowercase characters and a valid address.\n For more information access: https://dashboard.tailtarget.com/dmp/#/docs/a10");
        }


    }

    //To anonymize a E-Mail field, should use all lowercase characters (lower case).
    private boolean isValidEmail(String emailTxt){

        String email = emailTxt.trim().toLowerCase();

        //Log.d(TailDMP.TAG, email);

        if(email.length() >0){

            String emailPattern = "^[_A-Za-z0-9-]+(\\.[_A-Za-z0-9-]+)*@[A-Za-z0-9]+(\\.[A-Za-z0-9]+)*(\\.[A-Za-z]{2,})$";
            return email.matches(emailPattern);

        }else {
            return false;
        }
    }

    /**
     *
     * Set user data to be encrypted
     *
     *
      * @param cpfToHash
     */
    public void generateUserHashFromCPF(String cpfToHash)  {

        //clean spaces
        String cleanCPF = cpfToHash.trim();

        //remove all but digits
        String digitsOnly = cleanCPF.replaceAll("[^0-9]+","");
        //Log.d(TailDMP.TAG, digitsOnly);

        boolean isvalid = false;

        //must have 11 digits for CPF
        if(digitsOnly.length() == 11){
            //create hash
            isvalid = true;
        }else{
            isvalid = false;
        }

        if(isvalid){
            //To anonymize a CPF field, should only be used 11 numerical digits, without formatting (punctuation and strokes).
            deviceMapping.setUserHashCPF(digitsOnly);

        }else{
            Log.e(TailDMP.TAG, "Warning, Invalid format to generate a hash. CPF should only be used with 11 numerical digits .\n For more information access: https://dashboard.tailtarget.com/dmp/#/docs/a10");
        }
    }

    //To anonymize a CPF field, should only be used 11 numerical digits, without formatting (punctuation and strokes).
    private boolean isValidCPF(String cpf){

        //clean spaces
        String cleanCPF = cpf.trim();

        //remove all but digits
        String digitsOnly = cleanCPF.replaceAll("[^0-9]+","");
        //Log.d(TailDMP.TAG, digitsOnly);

        //must have 11 digits for CPF
        if(digitsOnly.length() == 11){
            //create hash
            return true;
        }else{
            return false;
        }

    }


    /**
     *
     * Set user data to be encrypted
     * To generate a hash from a phone use only phone numbers of its full international form ex: 11 91234-5678 = 55 11 91234-5678. Note that should be no international carrier codes in the phone's shape .
     * For more information access: https://dashboard.tailtarget.com/dmp/#/docs/a10
     *
     *
      * @param phoneToHash
     */
    public void generateUserHashFromPhone(String phoneToHash)  {

        //clean spaces
        String cleanTel = phoneToHash.trim();

        //remove all but digits
        String digitsOnly = cleanTel.replaceAll("[^0-9]+","");


        if(digitsOnly.length() <= 11){
            Log.w(TailDMP.TAG, "To generate a hash from a phone use only phone numbers of its full international form ex: Local phone number: 11 91234-5678 => International phone number: 55 11 91234-5678. \n Note that should be no international carrier codes in the phone's shape .\n For more information access: https://dashboard.tailtarget.com/dmp/#/docs/a10");

        }else{
            deviceMapping.setUserHashTel(onlyDigitsTel(phoneToHash));
        }

    }

    private String onlyDigitsTel(String tel){

        //clean spaces
        String cleanTel = tel.trim();

        //remove all but digits
        String digitsOnly = cleanTel.replaceAll("[^0-9]+","");

        return digitsOnly;

    }


    /**
     * Add a key/value pair to this user.
     *
     * @param key   - maximum of 32 characters
     * @param value - maximum of 32 characters
     * @return
     */
    private TailDMP addDmpValue(String key, String value) throws TailDMPException {
        if (!valid(key, 32) || !valid(value, 32)) {
            throw new TailDMPException(String.format("TailDMP: key and value can not be empty or exceed 32 characters. [%s] [%s] ", key, value));
        }

        this.dmpValues.valuesText.put(key, value);

        return this;
    }


    /**
     * Add a key/value pair to this user.
     *
     * @param key  - maximum of 32 characters
     * @param date - can not be null
     * @return
     */
    private TailDMP addDmpValue(String key, Date date) throws TailDMPException {
        if (!valid(key, 32) || date == null) {
            throw new TailDMPException(String.format("TailDMP: key and value can not be empty or exceed 32 characters. [%s] [%s] ", key, date));
        }

        DateFormat df = new DateFormat();
        df.format("yyyy-MM-ddTHH:mm:ssZ", date);

        this.dmpValues.valuesText.put(key, df.toString());

        return this;
    }


    /**
     * Add a key/value pair to this user.
     *
     * @param key   - maximum of 32 characters
     * @param value -  can not be null
     * @return
     */
    private TailDMP addDmpValue(String key, Boolean value) throws TailDMPException {
        if (!valid(key, 32) || value == null) {
            throw new TailDMPException(String.format("TailDMP: key and value can not be empty or exceed 32 characters. [%s] [%s] ", key, value));
        }

        dmpValues.valuesText.put(key, value.toString());

        return this;
    }


    /**
     * Add a key/value pair to this user.
     *
     * @param key   - maximum of 32 characters
     * @param value - can not be null
     * @return
     */
    private TailDMP addDmpValue(String key, Double value) throws TailDMPException {
        if (!valid(key, 32) || value == null) {
            throw new TailDMPException(String.format("TailDMP: key and value can not be empty or exceed 32 characters. [%s] [%s] ", key, value));
        }

        this.dmpValues.valuesText.put(key, value.toString());

        return this;
    }


    /**
     * Add a key/value pair to this user.
     *
     * @param key   - maximum of 32 characters
     * @param value - can not be null
     * @return
     */
    private TailDMP addDmpValue(String key, Long value) throws TailDMPException {
        if (!valid(key, 32) || value == null) {
            throw new TailDMPException(String.format("TailDMP: key and value can not be empty or exceed 32 characters. [%s] [%s] ", key, value));
        }

        this.dmpValues.valuesText.put(key, value.toString());

        return this;
    }


    /**
     * Add a key/values pair to this user.
     *
     * @param key    - maximum of 32 characters
     * @param values - can not be null or exceed 20 values
     * @return
     */
    private TailDMP addDmpValues(String key, String[] values) throws TailDMPException {
        if (!valid(key, 32) || !valid(values, 20, 32)) {
            throw new TailDMPException(String.format("TailDMP: key and value can not be empty, exceed 32 characters or exceed 20 values. [%s] [%s] ", key, values));
        }

        this.dmpValues.valuesTextList.put(key, new ArrayList<String>(Arrays.asList(values)));

        return this;
    }


    /**
     * Add a key/values pair to this user.
     *
     * @param key    - maximum of 32 characters
     * @param values - can not be null or exceed 20 values
     * @return
     */
    private TailDMP addDmpValues(String key, Long[] values) throws TailDMPException {
        if (!valid(key, 32) || values == null || values.length == 0 || values.length > 20) {
            throw new TailDMPException(String.format("TailDMP: key and value can not be empty, exceed 32 characters or exceed 20 values. [%s] [%s] ", key, values));
        }

        this.dmpValues.valuesLongList.put(key, new ArrayList<Long>(Arrays.asList(values)));

        return this;
    }


    /**
     * Set advertising id for this user
     * @param advertisingId
     * @return
     * @throws TailDMPException
     */
    public TailDMP setAdvertisingId(String advertisingId) throws TailDMPException {
        if (advertisingId == null || advertisingId.isEmpty()) {
            throw new TailDMPException("TailDMP: advertisingId can not be null or empty.");
        }

        this.dmpValues.advertisingId = advertisingId;

        return this;
    }


    private boolean valid(String[] values, int maxLenght, int maxSize) {
        if (values == null || values.length == 0 || values.length > maxLenght) {
            return false;
        }

        for (String value : values) {
            if (!valid(value, maxSize)) {
                return false;
            }
        }

        return true;
    }


    private boolean valid(String value, int maxSize) {
        return value != null && value.length() <= maxSize;
    }
//
//
//
//
//
//
//    public static void init(Context context) {
//
//        TailDMP.context = context;
//        TailDMP.config = new TailDMP_Config();
//        TailDMP.preferences = context.getSharedPreferences(TailDMP_Config.NAMESPACE, Context.MODE_PRIVATE);
//
//
//        try {
//            tailDMP_config.initialize(context);
//        } catch (TailDMPException e) {
//            e.printStackTrace();
//        }
//
//        tailDMPCrypto = new TailDMP_Crypto(context);
//
//
//        Log.d(TAG, "publicKey :" + tailDMPCrypto.encript("test"));
//
//
//        preferences.edit().putLong("lastUpdate", new Date().getTime()).commit();
//
//        System.out.println("lastUpdate: " + preferences.getLong("lastUpdate", 0l));
//
//    }


//    public static void showGooglePlayUpdateDialog(final Activity activity, DialogInterface.OnCancelListener cancelListener) {
//        final int googlePlayServicesCheck = GoogleApiAvailability.getInstance().isGooglePlayServicesAvailable(activity);
//        switch (googlePlayServicesCheck) {
//            case ConnectionResult.SUCCESS:
//                // PLay service updated and enabled, It is useless to display an update dialog.
//                break;
//            case ConnectionResult.SERVICE_DISABLED:
//            case ConnectionResult.SERVICE_INVALID:
//            case ConnectionResult.SERVICE_MISSING:
//            case ConnectionResult.SERVICE_VERSION_UPDATE_REQUIRED:
//                Dialog dialog = GoogleApiAvailability.getInstance().getErrorDialog(activity, 2, 0, cancelListener);
////                dialog.setOnCancelListener(new DialogInterface.OnCancelListener() {
////                    @Override
////                    public void onCancel(DialogInterface dialogInterface) {
////                        activity.finish();
////                    }
////                });
//                dialog.show();
//        }
//
//    }


    /**
     * Convert an input string to a respective lowercase SHA256 hash
     *
     * @param text input string
     * @return SHA256(text)
     */
    public static String convertToSHA256(String text) {
        if (text == null) {
            return null;
        }

        MessageDigest digest = null;
        try {
            digest = MessageDigest.getInstance("SHA-256");
        } catch (NoSuchAlgorithmException e1) {
            e1.printStackTrace();
            return null;
        }
        digest.reset();
        return bin2hex(digest.digest(text.getBytes())).toLowerCase();
    }

    /**
     * binary to hexadecimal helper
     *
     * @param data array of bytes
     * @return respective hexadecimal string
     */
    private static String bin2hex(byte[] data) {
        return String.format("%0" + (data.length * 2) + "X", new BigInteger(1, data));
    }

    private void getOpt(String data){

        if(data.equals("1")){
            this.optin = true;
        }else{
            this.optin = false;
        }
    }

    public boolean isOptin() {
        TailDMPDb db = TailDMPDb.getInstance(context);

        try{
            SQLiteDatabase dtbase = db.getdatabase();

            TAsyncGetUserData t = new TAsyncGetUserData(dtbase){

                @Override
                protected void onPostExecute(TailDMPData resultFromAsync) {
                    super.onPostExecute(resultFromAsync);

                    getOpt(resultFromAsync.optin);
                }
            };

        }catch (Exception e) {

            e.printStackTrace();
            Log.i(TailDMPValues.TAG,"We can't get this data, something wrong happened :( " );

        }finally {

            //Log.e(TAG,"OPDTIN ? "+this.optin);
            return this.optin;
        }
    }


    private void setOpt(boolean opt){
        this.optin = opt;
    }

    public void setOptin(boolean optin) {

        setOpt(optin);

        try {
            TailDMPDb db = TailDMPDb.getInstance(context);
            SQLiteDatabase dtbase = db.getdatabase();

            final boolean tmp = optin;

            TAsyncUpdateData t = new TAsyncUpdateData(dtbase) {

                @Override
                protected void onPostExecute(String s) {
                    super.onPostExecute(s);

                    if (s == "success") {
                        setOpt(tmp);
                    } else {
                        setOpt(false);
                    }
                }
            };


            int data = optin ? 1 : 0;

            Map<String, String> mp = new HashMap<String, String>();
            mp.put(TailDMPDb.DB_FIELD_OPTIN, data + "");

            //send data to be updated
            t.execute(mp);



        }catch(Exception e){

        }
    }

    public boolean isSendDataOnWifiOnly() {


        try{
            TailDMPDb db = TailDMPDb.getInstance(context);

            SQLiteDatabase dtbase = db.getdatabase();
            TAsyncGetUserData t = new TAsyncGetUserData(dtbase){

                @Override
                protected void onPostExecute(TailDMPData resultFromAsync) {
                    super.onPostExecute(resultFromAsync);

                    getSendDataOnWifiOnly(resultFromAsync.sendDataWifiOnly);
                }
            };

        }catch (Exception e) {

            e.printStackTrace();
            Log.i(TailDMPValues.TAG,"We can't get this data, something wrong happened :( " );

        }finally {
            return this.sendDataOnWifiOnly;
        }


        //return sendDataOnWifiOnly;
    }

    private void getSendDataOnWifiOnly(String data){

        if(data.equals("1")){
            this.sendDataOnWifiOnly = true;
        }else{
            this.sendDataOnWifiOnly = false;
        }


    }

    private void setWifiOnly(boolean data){
        this.sendDataOnWifiOnly = data;
    }

    public void setSendDataOnWifiOnly(boolean sendDataOnWifiOnly) {

        try {
            TailDMPDb db = TailDMPDb.getInstance(context);
            SQLiteDatabase dtbase = db.getdatabase();

            final boolean tmp = sendDataOnWifiOnly;

            TAsyncUpdateData t = new TAsyncUpdateData(dtbase) {

                @Override
                protected void onPostExecute(String s) {
                    super.onPostExecute(s);

                    if (s == "success") {
                        setWifiOnly(tmp);
                    } else {
                        setWifiOnly(false);
                    }
                }
            };

            int data = sendDataOnWifiOnly ? 1 : 0;

            Map<String, String> mp = new HashMap<String, String>();
            mp.put(TailDMPDb.DB_FIELD_SEND_DATA_WIFIONLY, data + "");

            //send data to be updated
            t.execute(mp);
        }catch(Exception e){
            e.printStackTrace();
        }

    }


//    public static void test(Application application) {
//        final PackageManager pm = application.getPackageManager();
//        //get a list of installed apps.
//        List<ApplicationInfo> packages = pm.getInstalledApplications(PackageManager.GET_META_DATA);
//
//        for (ApplicationInfo packageInfo : packages) {
//            Log.d(TAG, "Installed package :" + packageInfo.packageName);
//            Log.d(TAG, "Source dir : " + packageInfo.sourceDir);
//            Log.d(TAG, "Launch Activity :" + pm.getLaunchIntentForPackage(packageInfo.packageName));
//        }
//}



}
