/*
 * Copyright 2023 Nedra Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package digital.nedra.commons.starter.witsml.clt.transport;

import com.auth0.jwt.JWT;
import com.auth0.jwt.interfaces.DecodedJWT;
import java.time.Instant;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.ReentrantLock;
import lombok.RequiredArgsConstructor;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.stereotype.Component;

@Component
@ConditionalOnProperty(
    prefix = "starters.witsml-clt.witsml-server",
    name = "authentication-type",
    havingValue = "in-memory-bearer")
@RequiredArgsConstructor
public class InMemoryBearerAuthMessageSender extends AbstractBearerAuthMessageSender {

  private final ReentrantLock reentrantLock = new ReentrantLock();
  private final AtomicReference<DecodedJWT> jwtReference = new AtomicReference<>();
  private final TokenProvider tokenProvider;

  public String getToken() {
    if (isTokenExpired()) {
      reentrantLock.lock();
      try {
        if (isTokenExpired()) {
          DecodedJWT jwt = JWT.decode(tokenProvider.getAuthorizationToken());
          jwtReference.set(jwt);
        }
      } finally {
        reentrantLock.unlock();
      }
    }
    return jwtReference.get().getToken();
  }

  private boolean isTokenExpired() {
    DecodedJWT jwt = jwtReference.get();
    if (jwt == null) {
      return true;
    }
    Instant expirationDateTime = jwt.getExpiresAtAsInstant();
    return expirationDateTime == null || !Instant.now().isBefore(expirationDateTime);
  }
}

