/*
 * Copyright 2023 Nedra Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package digital.nedra.commons.starter.witsml.clt;

import digital.nedra.commons.starter.witsml.clt.config.WitsmlServerProperties;
import jakarta.annotation.PostConstruct;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.AutoConfigureOrder;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;
import org.springframework.oxm.jaxb.Jaxb2Marshaller;
import org.springframework.ws.transport.WebServiceMessageSender;
import org.springframework.ws.transport.http.HttpUrlConnectionMessageSender;

/**
 * Конфигурационный класс с настройками WITSML бинов.
 */
@Slf4j
@Configuration
@ComponentScan
@AutoConfigureOrder(Ordered.HIGHEST_PRECEDENCE)
@EnableConfigurationProperties(WitsmlServerProperties.class)
@RequiredArgsConstructor
public class WitsmlClientConfiguration {
  @PostConstruct
  public void init() {
    log.info("Starter: starters.witsml-clt started");
  }

  /**
   * Возвращает маршаллер отвечающий за сериализацию/десериализацию между XML и DTO объектами из
   * пакета digital.nedra.commons.starter.witsml.clt.model.soap.
   *
   * @return маршаллер DTO из пакета digital.nedra.commons.starter.witsml.clt.model.soap.
   */
  @Bean
  public Jaxb2Marshaller witsmlJaxb2Marshaller() {
    var jaxb2Marshaller = new Jaxb2Marshaller();
    jaxb2Marshaller.setContextPath("digital.nedra.commons.starter.witsml.clt.model.soap");
    return jaxb2Marshaller;
  }

  /**
   * Возвращает стандартный MessageSender при стандартном значении authentication-type.
   * @return HttpUrlConnectionMessageSender без внутреннего типа авторизации.
   */
  @Bean
  @ConditionalOnProperty(prefix = "starters.witsml-clt.witsml-server",
      name = "authentication-type",
      havingValue = "no-auth")
  public WebServiceMessageSender httpUrlConnectionMessageSender() {
    return new HttpUrlConnectionMessageSender();
  }

}