/*
 * Copyright 2022 Nedra Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package digital.nedra.commons.starter.swagger.config.security.schema;

import static digital.nedra.commons.starter.common.config.properties.SwaggerProperties.SecuritySchemaFlow.IMPLICIT;

import digital.nedra.commons.starter.common.config.properties.OAuth2Properties;
import digital.nedra.commons.starter.security.dto.OidcConfigurationDto;
import digital.nedra.commons.starter.security.service.OidcConfigurationResolver;
import digital.nedra.commons.starter.swagger.util.ScopeUtils;
import io.swagger.v3.oas.models.security.OAuthFlow;
import io.swagger.v3.oas.models.security.OAuthFlows;
import io.swagger.v3.oas.models.security.SecurityScheme;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;

@Component
public class ImplicitSchema extends CustomSecuritySchema {
  private static final String CONFIG_FAILED_MSG =
      "Unable to config implicit flow security schema. ";

  private final OAuth2Properties oauth2Properties;
  private final OidcConfigurationResolver oidcConfigurationResolver;
  private String authorizationUrl;

  protected ImplicitSchema(
      OAuth2Properties oauth2Properties,
      @Autowired(required = false) OidcConfigurationResolver oidcConfigurationResolver) {
    super(IMPLICIT);
    this.oauth2Properties = oauth2Properties;
    this.oidcConfigurationResolver = oidcConfigurationResolver;
  }

  @Override
  public SecurityScheme build() {
    setUp();
    return new SecurityScheme()
        .name("IMPLICIT")
        .type(SecurityScheme.Type.OAUTH2)
        .description("Oauth2 implicit flow")
        .in(SecurityScheme.In.HEADER)
        .flows(new OAuthFlows()
            .implicit(new OAuthFlow()
                .authorizationUrl(authorizationUrl)
                .scopes(ScopeUtils.buildScopesFromConfig(oauth2Properties.getScopes()))));
  }

  private void setUp() {
    final String authorizationUrlFromResource = oauth2Properties.getAuthorizationUrl();
    if (oidcConfigurationResolver != null) {
      this.authorizationUrl = oidcConfigurationResolver.getOidcConfiguration()
          .map(OidcConfigurationDto::getAuthorizationEndpoint)
          .orElse(authorizationUrlFromResource);
    } else {
      this.authorizationUrl = authorizationUrlFromResource;
    }

    Assert.notNull(authorizationUrl, CONFIG_FAILED_MSG + "Authorization url is null.");
    Assert.notNull(oauth2Properties.getClientId(), CONFIG_FAILED_MSG + "ClientId is null.");
  }
}
