/*
 * Copyright 2022 Nedra Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package digital.nedra.commons.starter.swagger.config;

import digital.nedra.commons.starter.common.config.properties.OAuth2Properties;
import digital.nedra.commons.starter.common.config.properties.SwaggerProperties;
import digital.nedra.commons.starter.swagger.config.security.schema.CustomSecuritySchema;
import io.swagger.v3.oas.models.Components;
import io.swagger.v3.oas.models.OpenAPI;
import io.swagger.v3.oas.models.security.SecurityRequirement;
import io.swagger.v3.oas.models.security.SecurityScheme;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.springdoc.core.models.GroupedOpenApi;
import org.springdoc.core.properties.SwaggerUiOAuthProperties;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;

@Configuration
@ConditionalOnClass(
    name = "digital.nedra.commons.starter.security.oauth2.bearer.OAuth2StarterAutoConfiguration")
@Slf4j
public class SwaggerOAuth2BearerConfigurer {
  private final OAuth2Properties oauth2Properties;
  private final SwaggerProperties swaggerProperties;
  private final Map<SwaggerProperties.SecuritySchemaFlow, CustomSecuritySchema>
      securitySchemasMap;
  private final OpenAPI openApiConfig;

  public SwaggerOAuth2BearerConfigurer(
      OAuth2Properties oauth2Properties,
      SwaggerProperties swaggerProperties,
      Set<CustomSecuritySchema> securitySchemeList) {
    this.oauth2Properties = oauth2Properties;
    this.swaggerProperties = swaggerProperties;
    this.securitySchemasMap = securitySchemeList.stream()
        .collect(Collectors.toMap(CustomSecuritySchema::getFlow, Function.identity()));
    this.openApiConfig = buildOpenApiConfig();
  }

  @Bean
  public OpenAPI getOpenApiConfig() {
    return this.openApiConfig;
  }

  @Bean
  public GroupedOpenApi adminApi() {
    return GroupedOpenApi.builder()
        .group("API")
        .pathsToMatch(swaggerProperties.getAllowedPaths())
        .build();
  }

  @Bean
  @Primary
  public SwaggerUiOAuthProperties swaggerUiOAuthProperties(SwaggerUiOAuthProperties properties) {
    properties.setClientId(oauth2Properties.getClientId());
    properties.setClientSecret(oauth2Properties.getClientSecret());
    properties.setScopeSeparator(" ");
    return properties;
  }

  private OpenAPI buildOpenApiConfig() {
    final List<SecurityScheme> securitySchemeList = getSelectedSchemes();
    final OpenAPI openApi = new OpenAPI();
    if (!securitySchemeList.isEmpty()) {
      final Components components = new Components();
      securitySchemeList.forEach(scheme -> components.addSecuritySchemes(scheme.getName(), scheme));
      final List<SecurityRequirement> securityRequirements = securitySchemeList.stream()
          .map(SecurityScheme::getName)
          .map(schema -> new SecurityRequirement().addList(schema, schema))
          .collect(Collectors.toList());
      openApi.components(components)
          .security(securityRequirements);

      securitySchemeList.forEach(
          scheme -> log.info("Security schema {} is added.", scheme.getName()));
    } else {
      log.warn("Oauth2 starter is enable, but swagger ui auth flow is empty.");
    }
    return openApi;
  }

  private List<SecurityScheme> getSelectedSchemes() {
    final List<SwaggerProperties.SecuritySchemaFlow> selectedFlows =
        swaggerProperties.getSecurity().getFlows();
    return selectedFlows.stream()
        .map(securitySchemasMap::get)
        .map(CustomSecuritySchema::build)
        .collect(Collectors.toList());
  }
}
