/*
 * Copyright 2022 Nedra Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package digital.nedra.commons.starter.security.engine.core;

import static java.util.Optional.ofNullable;
import static org.springframework.util.CollectionUtils.isEmpty;

import com.google.common.collect.ImmutableList;
import digital.nedra.commons.starter.security.engine.config.SecurityEngineProperties;
import digital.nedra.commons.starter.security.engine.config.SecurityEngineProperties.StartMode;
import digital.nedra.commons.starter.security.engine.core.dto.AuthorityDto;
import digital.nedra.commons.starter.security.engine.core.dto.AuthorityDto.RoleDto;
import digital.nedra.commons.starter.security.engine.core.dto.RoleAuthoritiesDto;
import digital.nedra.commons.starter.security.engine.core.dto.RoleAuthoritiesDto.RoleAuthorityDto;
import digital.nedra.commons.starter.security.engine.utils.SecurityUtils;
import jakarta.annotation.PostConstruct;
import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import javax.annotation.Nullable;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Component;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RestController;

@Slf4j
@Component
@RequiredArgsConstructor
public class SecurityEngine {

  private final List<ContextBuilder<? extends AuthorityContext>> builders;
  private final List<RoleHandler> handlers;

  private final RoleResolver roleResolver;
  private final SecurityEngineProperties securityEngineProperties;

  private ImmutableList<AuthorityWithContextHandler> authorityWithContextHandlers;

  @PostConstruct
  public void init() {
    checkIfAuthorityAssignedToAllUsersRoles();
  }

  public void checkPermission(final String authority,
                              final String description,
                              final Class<? extends ContextBuilder<AuthorityContext>> builder,
                              final AuthorityContextPayload payload) {

    final List<String> userRoles = roleResolver.getRoles();
    final List<RoleHandler> authHandlers = getPermissionHandlers(authority, userRoles);
    final boolean isNoPermissions = isEmpty(authHandlers);
    if (isNoPermissions) {
      throwAuthorityException(authority, description, userRoles);
    }
    final AuthorityContext context = createContext(builder, payload, userRoles);
    if (!isAllowed(authority, authHandlers, context)) {
      throwAuthorityException(authority, description, userRoles);
    }
  }

  public boolean hasPermission(
      final String authority,
      final Class<? extends ContextBuilder<AuthorityContext>> builder,
      final AuthorityContextPayload payload) {
    final List<String> userRoles = roleResolver.getRoles();
    final List<RoleHandler> authHandlers = getPermissionHandlers(authority, userRoles);
    if (isEmpty(authHandlers)) {
      return false;
    }
    final AuthorityContext context = this.createContext(builder, payload, userRoles);
    return isAllowed(authority, authHandlers, context);
  }

  public boolean isCalledFromRestController() {
    final String packageRegExFilter = securityEngineProperties.getRestControllerPackageFilter();
    final Pattern packagePattern = Pattern.compile(packageRegExFilter);
    return Stream.of(Thread.currentThread().getStackTrace())
        .map(StackTraceElement::getClassName)
        .filter(packagePattern.asMatchPredicate())
        .map(className -> {
          try {
            return Class.forName(className);
          } catch (ClassNotFoundException ex) {
            return null;
          }
        })
        .filter(Objects::nonNull)
        .anyMatch(clazz ->
            Objects.nonNull(clazz.getAnnotation(RestController.class))
                || Objects.nonNull(clazz.getAnnotation(Controller.class))
        );
  }

  public Set<String> getAllRoles() {
    return SecurityUtils.getAllRoles(this.handlers);
  }

  public List<String> getAllPermissions() {
    return SecurityUtils.getAllAuthorities(this.handlers)
        .stream()
        .sorted()
        .toList();
  }

  public List<String> getRolePermissions() {
    return SecurityUtils
        .getRoleAuthorities(roleResolver.getRoles(), handlers)
        .stream()
        .sorted()
        .toList();
  }

  public List<RoleAuthoritiesDto> getAllRoleAuthorities() {
    return this.handlers.stream()
        .map(handler -> {
          final List<RoleAuthorityDto> authoritiesList =
              handler.getAuthorities()
                  .stream()
                  .map(this::newRoleAuthorityDto)
                  .toList();
          return RoleAuthoritiesDto.builder()
              .role(handler.getRole())
              .authorities(ImmutableList.copyOf(authoritiesList))
              .build();
        })
        .toList();
  }

  public List<AuthorityDto> getAllAuthoritiesAndRoles() {
    final Set<String> allAuthorities = SecurityUtils.getAllAuthorities(handlers);
    return allAuthorities.stream()
        .distinct()
        .map(this::newAuthorities)
        .toList();
  }

  public void initAuthorityHandlerList(final ApplicationContext applicationContext)
      throws BeansException {
    final List<AuthorityWithContextHandler> authorityHandlerList =
        Stream.of(applicationContext.getBeanDefinitionNames())
            .filter(Predicate.not("securityEngineInitializer"::equals))
            .map(applicationContext::getBean)
            .map(this::findMethodsWithAuthorityCheckAnnotation)
            .flatMap(Collection::stream)
            .map(this::createAuthorityHandlerItem)
            .filter(Objects::nonNull)
            .toList();
    this.authorityWithContextHandlers = ImmutableList.copyOf(authorityHandlerList);
  }

  @SuppressWarnings("all")
  public List<String> calculateUserAuthoritiesForContextParameters(
      @Nullable final Map<String, Object> parameters) {

    Objects.requireNonNull(this.authorityWithContextHandlers);

    final List<String> availableAuthorities = getRolePermissions();

    final List<AuthorityWithContextHandler> authoritiesWithHandlers =
        this.authorityWithContextHandlers
            .stream()
            .filter(authority -> availableAuthorities.contains(authority.name()))
            .toList();

    final List<String> forbiddenAuthorities =
        authoritiesWithHandlers.stream()
            .filter(authority -> !isCurrentUserHasPermissionForAuthority(parameters, authority))
            .map(AuthorityWithContextHandler::name)
            .distinct()
            .toList();
    availableAuthorities.removeAll(forbiddenAuthorities);

    return availableAuthorities;
  }

  private AuthorityContext createContext(
      final Class<? extends ContextBuilder<AuthorityContext>> builder,
      final AuthorityContextPayload payload,
      final List<String> userRoles) {
    if (isBuilderNotSet(builder)) {
      return SecurityUtils.createDefaultContext(userRoles);
    }
    return this.builders
        .stream()
        .filter(builder::isInstance)
        .map(builder::cast)
        .map(authBuilder -> authBuilder.build(userRoles, payload))
        .findFirst()
        .orElseGet(() -> SecurityUtils.createDefaultContext(userRoles));
  }

  private boolean isBuilderNotSet(
      final Class<? extends ContextBuilder<AuthorityContext>> builder) {
    return ContextBuilder.class.equals(builder);
  }

  @SuppressWarnings("all")
  private List<RoleHandler> getPermissionHandlers(
      final String authority, final List<String> userRoles) {
    return this.handlers
        .stream()
        .filter(handler ->
            hasRole(userRoles, handler) && includeAuthority(authority, handler))
        .toList();
  }

  private boolean isAllowed(final String authority,
                            final List<RoleHandler> authHandlers,
                            final AuthorityContext context) {
    return authHandlers
        .stream()
        .anyMatch(handler -> handler.handle(authority, context));
  }

  private void throwAuthorityException(final String authority,
                                       final String description,
                                       final List<String> userRoles) {
    final String formatMessage = "Нет authority [%s] у роли [%s] для доступа к api %s.";
    throw new AuthorityException(formatMessage.formatted(authority, userRoles, description));
  }

  private boolean hasRole(final List<String> roles, final RoleHandler handler) {
    return ofNullable(handler.getRole())
        .map(r -> roles.stream().anyMatch(r::equalsIgnoreCase))
        .orElse(false);
  }

  private boolean includeAuthority(final String authority,
                                   final RoleHandler handler) {
    return handler.getAuthorities()
        .stream()
        .map(Authority::getName)
        .anyMatch(authority::equalsIgnoreCase);
  }

  private void checkIfAuthorityAssignedToAllUsersRoles() {
    final Map<String, Set<String>> authoritiesByRole =
        SecurityUtils.getRoleAuthorityMap(this.handlers);

    final Set<String> allUsedAuthorities =
        SecurityUtils.getAllUsedAuthorities(this.handlers);

    final StartMode startMode = securityEngineProperties.getStartMode();

    authoritiesByRole.entrySet()
        .stream()
        .filter(e -> e.getValue().size() < allUsedAuthorities.size())
        .forEach(e -> {
          final String name = e.getKey();
          final Set<String> roleAuths = e.getValue();
          final Set<String> a = new HashSet<>(allUsedAuthorities);
          a.removeAll(roleAuths);
          final String authRep = String.join(",", a);
          final String message =
              "Для роли [%s] не установлены права на [%s].".formatted(name, authRep);
          switch (startMode) {
            case STRICT -> throw new AuthorityException("Can't start security engine. " + message);
            case RELAXED -> log.warn(message);
            default -> log.warn("Security engine start mode=[{}] is not supported", startMode);
          }
        });
  }

  private RoleAuthorityDto newRoleAuthorityDto(final Authority auth) {
    final Optional<Fields> fields = ofNullable(auth.getContext());
    return RoleAuthorityDto.builder()
        .name(auth.getName())
        .fields(fields.map(Fields::getValue).orElse(null))
        .available(auth.isAvailable())
        .type(fields.map(Fields::getType).orElse(null))
        .build();
  }

  private AuthorityDto newAuthorities(final String authority) {
    final List<RoleDto> listOfRoles = this.handlers.stream()
        .map(handler -> handler.getAuthorities()
            .stream()
            .filter(auth -> authority.equalsIgnoreCase(auth.getName()))
            .findFirst()
            .map(auth -> newRole(handler, auth))
            .orElse(null))
        .filter(Objects::nonNull)
        .toList();

    return AuthorityDto.builder()
        .name(authority)
        .roles(ImmutableList.copyOf(listOfRoles))
        .build();
  }

  private RoleDto newRole(final RoleHandler handler,
                          final Authority authority) {
    final Optional<Fields> fields = ofNullable(authority.getContext());
    return RoleDto.builder()
        .name(handler.getRole())
        .available(authority.isAvailable())
        .fields(fields.map(Fields::getValue).orElse(null))
        .type(fields.map(Fields::getType).orElse(null))
        .build();
  }

  @SuppressWarnings("all")
  private boolean isCurrentUserHasPermissionForAuthority(
      final Map<String, Object> parameters,
      final AuthorityWithContextHandler authority) {

    final var defaultContextPayload = new DefaultContextPayload();
    if (Objects.nonNull(parameters)) {
      parameters.entrySet()
          .forEach(entry -> defaultContextPayload.set(entry.getKey(), entry.getValue()));
    }
    final Class<? extends ContextBuilder> clazz = authority.contextBuilder();
    return this.hasPermission(
        authority.name(),
        (Class<? extends ContextBuilder<AuthorityContext>>) clazz,
        defaultContextPayload);
  }

  private List<Method> findMethodsWithAuthorityCheckAnnotation(final Object bean) {
    final Class<?> clazz = bean.getClass();
    final Class<?> superClazz = clazz.getSuperclass();
    if (superClazz == null) {
      return Collections.emptyList();
    }
    final Method[] methods = superClazz.getDeclaredMethods();
    return Stream.of(methods)
        .filter(method -> {
          final AuthorityCheck authorityCheck = method.getAnnotation(AuthorityCheck.class);
          return authorityCheck != null && !ContextBuilder.class.equals(authorityCheck.handler());
        })
        .toList();
  }

  private AuthorityWithContextHandler createAuthorityHandlerItem(final Method method) {
    final AuthorityCheck annotation = method.getAnnotation(AuthorityCheck.class);
    final Annotation[][] parameterAnnotations = method.getParameterAnnotations();
    final List<String> authParamAnnotations = Stream.of(parameterAnnotations)
        .flatMap(Stream::of)
        .filter(parameterAnnotation ->
            AuthParam.class.equals(parameterAnnotation.annotationType()))
        .map(AuthParam.class::cast)
        .map(AuthParam::value)
        .filter(Objects::nonNull)
        .toList();
    return AuthorityWithContextHandler.builder()
        .contextBuilder(annotation.handler())
        .name(annotation.value())
        .parameters(ImmutableList.copyOf(authParamAnnotations))
        .build();
  }

}
