/*
 * Copyright 2022 Nedra Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package digital.nedra.commons.starter.security.engine.utils;

import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;

import digital.nedra.commons.starter.security.engine.core.Authority;
import digital.nedra.commons.starter.security.engine.core.AuthorityContext;
import digital.nedra.commons.starter.security.engine.core.DefaultContext;
import digital.nedra.commons.starter.security.engine.core.RoleHandler;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import lombok.experimental.UtilityClass;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.oauth2.jwt.Jwt;

@UtilityClass
public final class SecurityUtils {

  public static final String OID = "oid";

  public static Optional<String> getUserOid() {
    return getClaimValue(OID);
  }

  public static Optional<String> getClaimValue(final String propertyName) {
    return ofNullable(SecurityContextHolder.getContext().getAuthentication())
        .map(Authentication::getPrincipal)
        .filter(Jwt.class::isInstance)
        .map(Jwt.class::cast)
        .map(jwt -> jwt.getClaim(propertyName))
        .map(String.class::cast);
  }

  public static Map<String, Set<String>> getRoleAuthorityMap(final List<RoleHandler> handlers) {
    final List<String> roles = handlers
        .stream()
        .map(RoleHandler::getRole)
        .distinct()
        .collect(toList());
    final Map<String, Set<String>> map = new HashMap<>();
    for (String role : roles) {
      final Set<String> auth = handlers
          .stream()
          .filter(handler -> role.equalsIgnoreCase(handler.getRole()))
          .map(RoleHandler::getAuthorities)
          .flatMap(Collection::stream)
          .map(Authority::getName)
          .collect(toSet());
      map.put(role, auth);
    }
    return map;
  }

  public static Set<String> getAllUsedAuthorities(final List<RoleHandler> handlers) {
    return handlers
        .stream()
        .map(RoleHandler::getAuthorities)
        .flatMap(Collection::stream)
        .map(Authority::getName)
        .collect(toSet());
  }

  public static Set<String> getAllRoles(final List<RoleHandler> handlers) {
    final Map<String, Set<String>> map = getRoleAuthorityMap(handlers);
    return map.keySet();

  }

  public static Set<String> getAllAuthorities(final List<RoleHandler> handlers) {
    final Map<String, Set<String>> map = getRoleAuthorityMap(handlers);
    return map.values()
        .stream()
        .flatMap(Collection::stream)
        .collect(toSet());
  }

  public static List<String> getRoleAuthorities(final List<String> roles,
                                                final List<RoleHandler> handlers) {
    return handlers.stream()
        .filter(handler -> filterByRole(roles, handler))
        .map(SecurityUtils::getAuthorityList)
        .flatMap(Collection::stream)
        .distinct()
        .collect(toList());
  }

  public static AuthorityContext createDefaultContext(final List<String> roles) {
    final AuthorityContext defaultContext = new DefaultContext();
    defaultContext.setRoles(roles);
    return defaultContext;
  }

  private static boolean filterByRole(final List<String> roles, final RoleHandler handler) {
    return ofNullable(handler.getRole()).map(r -> roles.stream().anyMatch(r::equalsIgnoreCase))
        .orElse(false);
  }

  private static List<String> getAuthorityList(final RoleHandler handler) {
    return handler.getAuthorities().stream()
        .filter(Authority::isAvailable).map(Authority::getName).collect(toList());
  }

}
