/*
 * Copyright 2022 Nedra Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package digital.nedra.commons.starter.security.service;

import digital.nedra.commons.starter.common.config.properties.OAuth2Properties;
import digital.nedra.commons.starter.security.dto.OidcConfigurationDto;
import java.util.Optional;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.web.client.RestTemplateBuilder;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

@ConditionalOnClass(
    name = "digital.nedra.commons.starter.security.oauth2.bearer.OAuth2StarterAutoConfiguration")
@Service
public class OidcConfigurationResolver {
  private final RestTemplateBuilder restTemplateBuilder;
  private final Optional<OidcConfigurationDto> oidcConfiguration;

  public OidcConfigurationResolver(
      RestTemplateBuilder restTemplateBuilder,
      OAuth2Properties commonProperties) {
    this.restTemplateBuilder = restTemplateBuilder;
    this.oidcConfiguration = Optional.ofNullable(commonProperties.getIssuerUri())
        .map(this::resolve);
  }

  public Optional<OidcConfigurationDto> getOidcConfiguration() {
    return this.oidcConfiguration;
  }

  private OidcConfigurationDto resolve(String issuer) {
    RestTemplate rest = restTemplateBuilder.build();
    return rest.getForObject(
        issuer + "/.well-known/openid-configuration",
        OidcConfigurationDto.class);
  }
}
