/*
 * Copyright 2022 Nedra Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package digital.nedra.commons.starter.security.config;

import static digital.nedra.commons.starter.security.utils.Constants.API_PREFIX;

import digital.nedra.commons.starter.common.config.properties.SecurityProperties;
import digital.nedra.commons.starter.security.conditional.AnySecurityStarter;
import digital.nedra.commons.starter.security.utils.Constants;
import java.util.ArrayList;
import java.util.List;
import lombok.RequiredArgsConstructor;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Conditional;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.annotation.Order;
import org.springframework.security.config.Customizer;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configurers.AuthorizeHttpRequestsConfigurer;
import org.springframework.security.web.access.expression.WebExpressionAuthorizationManager;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;
import org.springframework.security.web.util.matcher.RequestMatcher;

@Conditional(AnySecurityStarter.class)
@Configuration
@RequiredArgsConstructor
public class SecurityMatchersConfig extends AbstractMatchersConfig {
  private final SecurityProperties securityProperties;

  @Bean
  @Order(100)
  public SessionMatchersConfiguration apiMatchers() {
    var antMatchers = new ArrayList<String>();
    antMatchers.add(Constants.API_PREFIX + "/**");
    String accessAttribute = "authenticated";
    return new SessionMatchersConfiguration(antMatchers, accessAttribute);
  }

  @Bean
  @Order(10)
  public SessionMatchersConfiguration sessionPublicMatchers() {
    var antMatchers = new ArrayList<String>();
    antMatchers.add(API_PREFIX + "/uaa/json");
    antMatchers.add(API_PREFIX + "/uaa/me");
    antMatchers.addAll(securityProperties.getIgnoredUrls());
    String accessAttribute = "permitAll";
    return new SessionMatchersConfiguration(antMatchers, accessAttribute);
  }

  @Bean
  public Customizer<AuthorizeHttpRequestsConfigurer<HttpSecurity>
      .AuthorizationManagerRequestMatcherRegistry> sessionAuthorizeRequestsCustomizer(
      List<SessionMatchersConfiguration> matchersConfigurations) {

    return requests ->
        matchersConfigurations.forEach(matchersConfiguration ->
            requests
                .requestMatchers(
                    matchersConfiguration.getAntMatchers().stream()
                        .<RequestMatcher>map(AntPathRequestMatcher::antMatcher)
                        .toArray(RequestMatcher[]::new))
                .access(new WebExpressionAuthorizationManager(matchersConfiguration.getSpelExp()))
        );
  }

}
