/*
 * Copyright 2022 Nedra Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package digital.nedra.commons.starter.keycloak.session.config;

import static digital.nedra.commons.starter.security.config.RequestMatcherUtil.toRequestMatcher;
import static java.util.Objects.nonNull;

import digital.nedra.commons.starter.keycloak.session.config.support.KeycloakLogoutHandler;
import digital.nedra.commons.starter.keycloak.session.config.support.KeycloakOauth2UserService;
import digital.nedra.commons.starter.keycloak.session.config.support.XhrCookieRequestCache;
import digital.nedra.commons.starter.security.config.BasicMatchersConfiguration;
import digital.nedra.commons.starter.security.config.SessionMatchersConfiguration;
import java.util.ArrayList;
import java.util.List;
import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.annotation.Order;
import org.springframework.security.config.Customizer;
import org.springframework.security.config.annotation.method.configuration.EnableMethodSecurity;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configurers.AbstractHttpConfigurer;
import org.springframework.security.config.annotation.web.configurers.AuthorizeHttpRequestsConfigurer;
import org.springframework.security.config.annotation.web.configurers.ExceptionHandlingConfigurer;
import org.springframework.security.web.SecurityFilterChain;

@Configuration
@EnableWebSecurity
@EnableMethodSecurity
@RequiredArgsConstructor
public class SecurityConfigurerAdapter {

  private final KeycloakLogoutHandler logoutHandler;
  private final XhrCookieRequestCache requestCache;
  private final KeycloakOauth2UserService keycloakOidcUserService;
  private final Customizer<AuthorizeHttpRequestsConfigurer<HttpSecurity>
      .AuthorizationManagerRequestMatcherRegistry> sessionAuthorizeRequestsCustomizer;
  private final Customizer<AuthorizeHttpRequestsConfigurer<HttpSecurity>
      .AuthorizationManagerRequestMatcherRegistry> basicAuthorizeRequestsCustomizer;
  @Value("${spring.security.oauth2.client.registration.sso.provider}")
  private String oauth2Provider;

  @Autowired(required = false)
  private Customizer<ExceptionHandlingConfigurer<HttpSecurity>> exceptionHandlingCustomizer;
  private final List<BasicMatchersConfiguration> basicMatchersConfigurations;
  private final List<SessionMatchersConfiguration> sessionMatchersConfigurations;

  @Bean
  @Order(10)
  @ConditionalOnBean(BasicMatchersConfiguration.class)
  public SecurityFilterChain basicFilterChain(HttpSecurity http) throws Exception {
    if (nonNull(exceptionHandlingCustomizer)) {
      http.exceptionHandling(exceptionHandlingCustomizer);
    }

    return http
        .csrf(AbstractHttpConfigurer::disable)
        .securityMatcher(toRequestMatcher(basicMatchersConfigurations))
        .authorizeHttpRequests(basicAuthorizeRequestsCustomizer)
        .httpBasic(Customizer.withDefaults())
        .build();
  }

  @Bean
  @Order(30)
  @ConditionalOnBean(SessionMatchersConfiguration.class)
  public SecurityFilterChain sessionFilterChain(HttpSecurity http) throws Exception {
    if (nonNull(exceptionHandlingCustomizer)) {
      http.exceptionHandling(exceptionHandlingCustomizer);
    }
    List<SessionMatchersConfiguration> matchersConfigurations =
        new ArrayList<>(sessionMatchersConfigurations);
    matchersConfigurations
        .add(new SessionMatchersConfiguration(List.of("/oauth2/**", "/login/**"), null));

    return http
        .securityMatcher(toRequestMatcher(matchersConfigurations))
        .authorizeHttpRequests(sessionAuthorizeRequestsCustomizer)
        .oauth2Login(oauth2Login ->
            oauth2Login
                .loginPage("/oauth2/authorization/" + oauth2Provider)
                .userInfoEndpoint(userInfoEndpoint ->
                    userInfoEndpoint.oidcUserService(keycloakOidcUserService)
                )
        )
        .oauth2Client(Customizer.withDefaults())
        .logout(logout -> logout.addLogoutHandler(logoutHandler))
        .csrf(AbstractHttpConfigurer::disable)
        .cors(Customizer.withDefaults())
        .requestCache(cache -> cache.requestCache(requestCache))
        .build();
  }

}
