/*
 * Copyright 2022 Nedra Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package digital.nedra.commons.starter.keycloak.session.config;

import digital.nedra.commons.starter.security.config.SessionMatchersConfiguration;
import java.util.ArrayList;
import java.util.List;
import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.security.oauth2.client.OAuth2ClientProperties;
import org.springframework.boot.autoconfigure.security.oauth2.client.OAuth2ClientPropertiesMapper;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.security.oauth2.client.InMemoryOAuth2AuthorizedClientService;
import org.springframework.security.oauth2.client.OAuth2AuthorizedClientService;
import org.springframework.security.oauth2.client.registration.ClientRegistrationRepository;
import org.springframework.security.oauth2.client.registration.InMemoryClientRegistrationRepository;
import org.springframework.security.oauth2.jwt.JwtDecoder;
import org.springframework.security.oauth2.jwt.NimbusJwtDecoder;

@Configuration
@RequiredArgsConstructor
public class SecurityConfig {

  @Value("${spring.security.oauth2.client.registration.sso.provider}")
  private String oauth2Provider;

  @Bean
  public SessionMatchersConfiguration swaggerMatchers() {
    return new SessionMatchersConfiguration(List.of("/swagger-ui/**"), "authenticated");
  }

  @Bean
  public SessionMatchersConfiguration logoutMatcher() {
    return new SessionMatchersConfiguration(List.of("/logout"), "authenticated");
  }

  @Bean
  @ConditionalOnMissingBean(JwtDecoder.class)
  public JwtDecoder keycloakJwtDecoder(OAuth2ClientProperties oauth2ClientProperties) {
    return NimbusJwtDecoder
        .withJwkSetUri(oauth2ClientProperties.getProvider().get(oauth2Provider).getJwkSetUri())
        .build();
  }

  /**
   * Please use JdbcOAuth2AuthorizedClientService and Spring Sessions
   * in case when you have multiple backend instances.
   */
  @Bean
  @ConditionalOnMissingBean(OAuth2AuthorizedClientService.class)
  public OAuth2AuthorizedClientService authorizedClientService(
      ClientRegistrationRepository clientRegistrationRepository) {
    return new InMemoryOAuth2AuthorizedClientService(clientRegistrationRepository);
  }

  @Bean
  public ClientRegistrationRepository clientRegistrationRepository(
      OAuth2ClientProperties properties) {

    var mapper = new OAuth2ClientPropertiesMapper(properties);
    var registrations = new ArrayList<>(mapper.asClientRegistrations().values());
    return new InMemoryClientRegistrationRepository(registrations);
  }
}
