/*
 * Copyright 2022 Nedra Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package digital.nedra.commons.starter.keycloak.session.config.support;

import static digital.nedra.commons.starter.keycloak.session.config.support.XhrRedirectFilter.XML_HTTP_REQUEST;
import static digital.nedra.commons.starter.keycloak.session.config.support.XhrRedirectFilter.X_REQUESTED_WITH;

import jakarta.servlet.http.Cookie;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import java.util.Base64;
import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.security.web.savedrequest.CookieRequestCache;
import org.springframework.security.web.util.UrlUtils;
import org.springframework.security.web.util.matcher.AnyRequestMatcher;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

@RequiredArgsConstructor
@Component
public class XhrCookieRequestCache extends CookieRequestCache {
  @Value("${starters.keycloak.redirect-mode}")
  private String redirectMode;

  @Override
  public void saveRequest(HttpServletRequest request, HttpServletResponse response) {
    if (!AnyRequestMatcher.INSTANCE.matches(request)) {
      this.logger.debug("Request not saved as configured RequestMatcher did not match");
      return;
    }
    String redirectUrl = getRedirectUrl(request);

    Cookie savedCookie =
        new Cookie("REDIRECT_URI", Base64.getEncoder().encodeToString(redirectUrl.getBytes()));
    savedCookie.setMaxAge(-1);
    savedCookie.setSecure(request.isSecure());
    String contextPath = request.getContextPath();
    String cookiePath = (!StringUtils.isEmpty(contextPath)) ? contextPath : "/";
    savedCookie.setPath(cookiePath);
    savedCookie.setHttpOnly(true);
    response.addCookie(savedCookie);
  }

  private String getRedirectUrl(HttpServletRequest request) {
    if (XhrRedirectFilter.RedirectMode.ALWAYS_REDIRECT.name().equals(redirectMode)) {
      return UrlUtils.buildFullRequestUrl(request);
    } else if (XhrRedirectFilter.RedirectMode.HEADER_BASED.name().equals(redirectMode)) {
      if (XML_HTTP_REQUEST.equals(request.getHeader(X_REQUESTED_WITH))) {
        return request.getHeader("Referer");
      } else {
        return UrlUtils.buildFullRequestUrl(request);
      }
    }
    throw new RuntimeException("Unsupported redirect mode: " + redirectMode);
  }
}
