/*
 * Copyright 2022 Nedra Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package digital.nedra.commons.starter.keycloak.session.config;

import static digital.nedra.commons.starter.security.config.RequestMatcherUtil.toRequestMatcher;
import static java.util.Objects.nonNull;

import digital.nedra.commons.starter.keycloak.session.config.support.KeycloakLogoutHandler;
import digital.nedra.commons.starter.keycloak.session.config.support.KeycloakOauth2UserService;
import digital.nedra.commons.starter.keycloak.session.config.support.XhrCookieRequestCache;
import digital.nedra.commons.starter.security.config.BasicMatchersConfiguration;
import digital.nedra.commons.starter.security.config.SessionMatchersConfiguration;
import java.util.ArrayList;
import java.util.List;
import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.annotation.Order;
import org.springframework.security.config.Customizer;
import org.springframework.security.config.annotation.method.configuration.EnableGlobalMethodSecurity;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configurers.ExceptionHandlingConfigurer;
import org.springframework.security.config.annotation.web.configurers.ExpressionUrlAuthorizationConfigurer;
import org.springframework.security.web.SecurityFilterChain;

@EnableWebSecurity
@EnableGlobalMethodSecurity(prePostEnabled = true)
@Configuration
@RequiredArgsConstructor
public class SecurityConfigurerAdapter {
  private final KeycloakLogoutHandler logoutHandler;
  private final XhrCookieRequestCache requestCache;
  private final KeycloakOauth2UserService keycloakOidcUserService;
  private final Customizer<ExpressionUrlAuthorizationConfigurer<HttpSecurity>
      .ExpressionInterceptUrlRegistry> sessionAuthorizeRequestsCustomizer;
  private final Customizer<ExpressionUrlAuthorizationConfigurer<HttpSecurity>
      .ExpressionInterceptUrlRegistry> basicAuthorizeRequestsCustomizer;
  @Value("${spring.security.oauth2.client.registration.sso.provider}")
  private String oauth2Provider;

  @Autowired(required = false)
  private Customizer<ExceptionHandlingConfigurer<HttpSecurity>> exceptionHandlingCustomizer;
  private final List<BasicMatchersConfiguration> basicMatchersConfigurations;
  private final List<SessionMatchersConfiguration> sessionMatchersConfigurations;

  @Bean
  @Order(10)
  @ConditionalOnBean(BasicMatchersConfiguration.class)
  public SecurityFilterChain basicFilterChain(HttpSecurity http) throws Exception {
    if (nonNull(exceptionHandlingCustomizer)) {
      http.exceptionHandling(exceptionHandlingCustomizer);
    }
    http.csrf().disable()
        .securityMatcher(toRequestMatcher(basicMatchersConfigurations))
        .authorizeRequests(basicAuthorizeRequestsCustomizer)
        .httpBasic();
    return http.build();
  }

  @Bean
  @Order(30)
  @ConditionalOnBean(SessionMatchersConfiguration.class)
  public SecurityFilterChain sessionFilterChain(HttpSecurity http) throws Exception {
    if (nonNull(exceptionHandlingCustomizer)) {
      http.exceptionHandling(exceptionHandlingCustomizer);
    }
    List<SessionMatchersConfiguration> matchersConfigurations =
        new ArrayList<>(sessionMatchersConfigurations);
    matchersConfigurations
        .add(new SessionMatchersConfiguration(List.of("/oauth2/**","/login/**"), null));

    return http
        .securityMatcher(toRequestMatcher(matchersConfigurations))
        .authorizeRequests(sessionAuthorizeRequestsCustomizer)
        .oauth2Login(httpSec -> httpSec
            .loginPage("/oauth2/authorization/" + oauth2Provider)
            .userInfoEndpoint()
            .oidcUserService(keycloakOidcUserService))
        .oauth2Client()
        .and()
        .logout()
        .addLogoutHandler(logoutHandler)
        .and()
        .csrf().disable()
        .cors()
        .and()
        .requestCache(i -> i.requestCache(requestCache))
        .build();
  }

}
